# frozen_string_literal: true
#
# Copyright:: 2019, Chef Software, Inc.
# Author:: Tim Smith (<tsmith84@gmail.com>)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
module RuboCop
  module Cop
    module Chef
      module RedundantCode
        # There is no need to notify an apt-get update when an apt_repository is created as this is done automatically by the apt_repository resource.
        #
        # @example
        #
        #   ### incorrect
        #   apt_repository 'my repo' do
        #     uri 'http://packages.example.com/debian'
        #     components %w(stable main)
        #     deb_src false
        #     notifies :run, 'execute[apt-get update]', :immediately
        #   end
        #
        #   ### correct
        #   apt_repository 'my repo' do
        #     uri 'http://packages.example.com/debian'
        #     components %w(stable main)
        #     deb_src false
        #   end
        #
        class AptRepositoryNotifiesAptUpdate < Base
          include RuboCop::Chef::CookbookHelpers
          include RangeHelp
          extend AutoCorrector

          MSG = 'There is no need to notify an apt-get update when an apt_repository is created as this is done automatically by the apt_repository resource.'

          def on_block(node)
            match_property_in_resource?(:apt_repository, 'notifies', node) do |notifies|
              return unless notifies.arguments[1] == s(:str, 'execute[apt-get update]')
              add_offense(notifies, severity: :refactor) do |corrector|
                corrector.remove(range_with_surrounding_space(range: notifies.loc.expression, side: :left))
              end
            end
          end
        end
      end
    end
  end
end
