# frozen_string_literal: true

#
# Author:: Greg Fitzgerald (<greg@gregf.org>)
#
# Copyright (C) 2013, Greg Fitzgerald
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

require "benchmark" unless defined?(Benchmark)
require "droplet_kit" unless defined?(DropletKit)
require "kitchen"
require "etc" unless defined?(Etc)
require "socket" unless defined?(Socket)
require "json" unless defined?(JSON)
autoload :YAML, "yaml"

module Kitchen
  module Driver
    # Digital Ocean driver for Kitchen.
    #
    # @author Greg Fitzgerald <greg@gregf.org>
    class Digitalocean < Kitchen::Driver::SSHBase
      default_config :username, "root"
      default_config :port, "22"
      default_config :size, "s-1vcpu-1gb"
      default_config :monitoring, false
      default_config(:image, &:default_image)
      default_config(:server_name, &:default_name)
      default_config :private_networking, true
      default_config :ipv6, false
      default_config :user_data, nil
      default_config :tags, nil
      default_config :firewalls, nil
      default_config :vpcs, nil

      default_config :api_url do
        ENV["DIGITALOCEAN_API_URL"] || "https://api.digitalocean.com"
      end

      default_config :region do
        ENV["DIGITALOCEAN_REGION"] || "nyc1"
      end

      default_config :digitalocean_access_token do
        ENV["DIGITALOCEAN_ACCESS_TOKEN"]
      end

      default_config :ssh_key_ids do
        ENV["DIGITALOCEAN_SSH_KEY_IDS"] || ENV["SSH_KEY_IDS"]
      end

      required_config :digitalocean_access_token
      required_config :ssh_key_ids

      PLATFORM_SLUG_MAP = {
        "centos-7" => "centos-7-x64",
        "centos-8" => "centos-8-x64",
        "debian-9" => "debian-9-x64",
        "debian-10" => "debian-10-x64",
        "debian-11" => "debian-11-x64",
        "fedora-32" => "fedora-32-x64",
        "fedora-33" => "fedora-33-x64",
        "freebsd-11" => "freebsd-11-x64-zfs",
        "freebsd-12" => "freebsd-12-x64-zfs",
        "ubuntu-16" => "ubuntu-16-04-x64",
        "ubuntu-18" => "ubuntu-18-04-x64",
        "ubuntu-20" => "ubuntu-20-04-x64",
      }.freeze

      def create(state)
        server = create_server

        state[:server_id] = server.id

        info("DigitalOcean instance <#{state[:server_id]}> created.")

        loop do
          sleep 8
          droplet = client.droplets.find(id: state[:server_id])

          break if droplet && droplet.networks[:v4] &&
            droplet.networks[:v4].any? { |n| n[:type] == "public" }
        end
        droplet ||= client.droplets.find(id: state[:server_id])

        state[:hostname] = droplet.networks[:v4]
          .find { |n| n[:type] == "public" }["ip_address"]

        if config[:firewalls]
          debug("trying to add the firewall by id")
          fw_ids = if config[:firewalls].is_a?(String)
                     config[:firewalls].split(/\s+|,\s+|,+/)
                   elsif config[:firewalls].is_a?(Array)
                     config[:firewalls]
                   else
                     warn("firewalls attribute is not string/array, ignoring")
                     []
                   end
          debug("firewall : #{YAML.dump(fw_ids.inspect)}")
          fw_ids.each do |fw_id|
            firewall = client.firewalls.find(id: fw_id)
            if firewall
              client.firewalls
                .add_droplets([droplet.id], id: firewall.id)
              debug("firewall added: #{firewall.id}")
            else
              warn("firewalls id: '#{fw_id}' was not found in api, ignoring")
            end
          end
        end

        wait_for_sshd(state[:hostname]); print "(ssh ready)\n"
        debug("digitalocean:create #{state[:hostname]}")
      end

      def destroy(state)
        return if state[:server_id].nil?

        # A new droplet cannot be destroyed before it is active
        # Retry destroying the droplet as long as its status is "new"
        loop do
          droplet = client.droplets.find(id: state[:server_id])

          break unless droplet

          if droplet.status != "new"
            client.droplets.delete(id: state[:server_id])
            break
          end

          info("Waiting on DigitalOcean instance <#{state[:server_id]}>
               to be active to destroy it, retrying in 8 seconds")
          sleep 8
        end

        info("DigitalOcean instance <#{state[:server_id]}> destroyed.")
        state.delete(:server_id)
        state.delete(:hostname)
      end

      # This method attempts to fetch the platform from a list of well-known
      # platform => slug mappings, and falls back to using just the platform as
      # provided if it can't find a mapping.
      def default_image
        PLATFORM_SLUG_MAP.fetch(instance.platform.name,
                                instance.platform.name)
      end

      # Generate what should be a unique server name up to 63 total chars
      # Base name:    15
      # Username:     15
      # Hostname:     23
      # Random string: 7
      # Separators:    3
      # ================
      # Total:        63
      def default_name
        [
          instance.name.gsub(/\W/, "")[0..14],
          (Etc.getlogin || "nologin").gsub(/\W/, "")[0..14],
          Socket.gethostname.gsub(/\W/, "")[0..22],
          Array.new(7) { rand(36).to_s(36) }.join,
        ].join("-").tr("_", "-")
      end

      private

      def client
        debug_client_config

        DropletKit::Client.new(access_token: config[:digitalocean_access_token], api_url: config[:api_url])
      end

      def create_server
        debug_server_config

        droplet = DropletKit::Droplet.new(
          name: config[:server_name],
          region: config[:region],
          image: config[:image],
          size: config[:size],
          monitoring: config[:monitoring],
          ssh_keys: config[:ssh_key_ids].to_s.split(/, ?/),
          private_networking: config[:private_networking],
          ipv6: config[:ipv6],
          user_data: config[:user_data],
          vpc_uuid: config[:vpcs],
          tags: if config[:tags].is_a?(String)
                  config[:tags].split(/\s+|,\s+|,+/)
                else
                  config[:tags]
                end
        )

        resp = client.droplets.create(droplet)

        if resp.class != DropletKit::Droplet
          error JSON.parse(resp)["message"]
          error "Please check your access token is set correctly."
        else
          resp
        end
      end

      def debug_server_config
        debug("digitalocean:name #{config[:server_name]}")
        debug("digitalocean:image#{config[:image]}")
        debug("digitalocean:size #{config[:size]}")
        debug("digitalocean:monitoring #{config[:monitoring]}")
        debug("digitalocean:region #{config[:region]}")
        debug("digitalocean:ssh_key_ids #{config[:ssh_key_ids]}")
        debug("digitalocean:private_networking #{config[:private_networking]}")
        debug("digitalocean:ipv4 #{config[:ipv4]})")
        debug("digitalocean:ipv6 #{config[:ipv6]}")
        debug("digitalocean:user_data #{config[:user_data]}")
        debug("digitalocean:tags #{config[:tags]}")
        debug("digitalocean:firewalls #{config[:firewalls]}")
      end

      def debug_client_config
        debug("digitalocean_api_key #{config[:digitalocean_access_token]}")
      end
    end
  end
end

# vim: ai et ts=2 sts=2 sw=2 ft=ruby
