# Copyright (C) 2018-2020 MongoDB Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

module Mongo
  module Auth
    module StringPrep
      # Contains character tables defined by RFC 3454 (string preparation).
      #
      # @since 2.6.0
      # @api private
      module Tables
        # Table A1 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        A1 = [
          0x0221..0x0221,
          0x0234..0x024F,
          0x02AE..0x02AF,
          0x02EF..0x02FF,
          0x0350..0x035F,
          0x0370..0x0373,
          0x0376..0x0379,
          0x037B..0x037D,
          0x037F..0x0383,
          0x038B..0x038B,
          0x038D..0x038D,
          0x03A2..0x03A2,
          0x03CF..0x03CF,
          0x03F7..0x03FF,
          0x0487..0x0487,
          0x04CF..0x04CF,
          0x04F6..0x04F7,
          0x04FA..0x04FF,
          0x0510..0x0530,
          0x0557..0x0558,
          0x0560..0x0560,
          0x0588..0x0588,
          0x058B..0x0590,
          0x05A2..0x05A2,
          0x05BA..0x05BA,
          0x05C5..0x05CF,
          0x05EB..0x05EF,
          0x05F5..0x060B,
          0x060D..0x061A,
          0x061C..0x061E,
          0x0620..0x0620,
          0x063B..0x063F,
          0x0656..0x065F,
          0x06EE..0x06EF,
          0x06FF..0x06FF,
          0x070E..0x070E,
          0x072D..0x072F,
          0x074B..0x077F,
          0x07B2..0x0900,
          0x0904..0x0904,
          0x093A..0x093B,
          0x094E..0x094F,
          0x0955..0x0957,
          0x0971..0x0980,
          0x0984..0x0984,
          0x098D..0x098E,
          0x0991..0x0992,
          0x09A9..0x09A9,
          0x09B1..0x09B1,
          0x09B3..0x09B5,
          0x09BA..0x09BB,
          0x09BD..0x09BD,
          0x09C5..0x09C6,
          0x09C9..0x09CA,
          0x09CE..0x09D6,
          0x09D8..0x09DB,
          0x09DE..0x09DE,
          0x09E4..0x09E5,
          0x09FB..0x0A01,
          0x0A03..0x0A04,
          0x0A0B..0x0A0E,
          0x0A11..0x0A12,
          0x0A29..0x0A29,
          0x0A31..0x0A31,
          0x0A34..0x0A34,
          0x0A37..0x0A37,
          0x0A3A..0x0A3B,
          0x0A3D..0x0A3D,
          0x0A43..0x0A46,
          0x0A49..0x0A4A,
          0x0A4E..0x0A58,
          0x0A5D..0x0A5D,
          0x0A5F..0x0A65,
          0x0A75..0x0A80,
          0x0A84..0x0A84,
          0x0A8C..0x0A8C,
          0x0A8E..0x0A8E,
          0x0A92..0x0A92,
          0x0AA9..0x0AA9,
          0x0AB1..0x0AB1,
          0x0AB4..0x0AB4,
          0x0ABA..0x0ABB,
          0x0AC6..0x0AC6,
          0x0ACA..0x0ACA,
          0x0ACE..0x0ACF,
          0x0AD1..0x0ADF,
          0x0AE1..0x0AE5,
          0x0AF0..0x0B00,
          0x0B04..0x0B04,
          0x0B0D..0x0B0E,
          0x0B11..0x0B12,
          0x0B29..0x0B29,
          0x0B31..0x0B31,
          0x0B34..0x0B35,
          0x0B3A..0x0B3B,
          0x0B44..0x0B46,
          0x0B49..0x0B4A,
          0x0B4E..0x0B55,
          0x0B58..0x0B5B,
          0x0B5E..0x0B5E,
          0x0B62..0x0B65,
          0x0B71..0x0B81,
          0x0B84..0x0B84,
          0x0B8B..0x0B8D,
          0x0B91..0x0B91,
          0x0B96..0x0B98,
          0x0B9B..0x0B9B,
          0x0B9D..0x0B9D,
          0x0BA0..0x0BA2,
          0x0BA5..0x0BA7,
          0x0BAB..0x0BAD,
          0x0BB6..0x0BB6,
          0x0BBA..0x0BBD,
          0x0BC3..0x0BC5,
          0x0BC9..0x0BC9,
          0x0BCE..0x0BD6,
          0x0BD8..0x0BE6,
          0x0BF3..0x0C00,
          0x0C04..0x0C04,
          0x0C0D..0x0C0D,
          0x0C11..0x0C11,
          0x0C29..0x0C29,
          0x0C34..0x0C34,
          0x0C3A..0x0C3D,
          0x0C45..0x0C45,
          0x0C49..0x0C49,
          0x0C4E..0x0C54,
          0x0C57..0x0C5F,
          0x0C62..0x0C65,
          0x0C70..0x0C81,
          0x0C84..0x0C84,
          0x0C8D..0x0C8D,
          0x0C91..0x0C91,
          0x0CA9..0x0CA9,
          0x0CB4..0x0CB4,
          0x0CBA..0x0CBD,
          0x0CC5..0x0CC5,
          0x0CC9..0x0CC9,
          0x0CCE..0x0CD4,
          0x0CD7..0x0CDD,
          0x0CDF..0x0CDF,
          0x0CE2..0x0CE5,
          0x0CF0..0x0D01,
          0x0D04..0x0D04,
          0x0D0D..0x0D0D,
          0x0D11..0x0D11,
          0x0D29..0x0D29,
          0x0D3A..0x0D3D,
          0x0D44..0x0D45,
          0x0D49..0x0D49,
          0x0D4E..0x0D56,
          0x0D58..0x0D5F,
          0x0D62..0x0D65,
          0x0D70..0x0D81,
          0x0D84..0x0D84,
          0x0D97..0x0D99,
          0x0DB2..0x0DB2,
          0x0DBC..0x0DBC,
          0x0DBE..0x0DBF,
          0x0DC7..0x0DC9,
          0x0DCB..0x0DCE,
          0x0DD5..0x0DD5,
          0x0DD7..0x0DD7,
          0x0DE0..0x0DF1,
          0x0DF5..0x0E00,
          0x0E3B..0x0E3E,
          0x0E5C..0x0E80,
          0x0E83..0x0E83,
          0x0E85..0x0E86,
          0x0E89..0x0E89,
          0x0E8B..0x0E8C,
          0x0E8E..0x0E93,
          0x0E98..0x0E98,
          0x0EA0..0x0EA0,
          0x0EA4..0x0EA4,
          0x0EA6..0x0EA6,
          0x0EA8..0x0EA9,
          0x0EAC..0x0EAC,
          0x0EBA..0x0EBA,
          0x0EBE..0x0EBF,
          0x0EC5..0x0EC5,
          0x0EC7..0x0EC7,
          0x0ECE..0x0ECF,
          0x0EDA..0x0EDB,
          0x0EDE..0x0EFF,
          0x0F48..0x0F48,
          0x0F6B..0x0F70,
          0x0F8C..0x0F8F,
          0x0F98..0x0F98,
          0x0FBD..0x0FBD,
          0x0FCD..0x0FCE,
          0x0FD0..0x0FFF,
          0x1022..0x1022,
          0x1028..0x1028,
          0x102B..0x102B,
          0x1033..0x1035,
          0x103A..0x103F,
          0x105A..0x109F,
          0x10C6..0x10CF,
          0x10F9..0x10FA,
          0x10FC..0x10FF,
          0x115A..0x115E,
          0x11A3..0x11A7,
          0x11FA..0x11FF,
          0x1207..0x1207,
          0x1247..0x1247,
          0x1249..0x1249,
          0x124E..0x124F,
          0x1257..0x1257,
          0x1259..0x1259,
          0x125E..0x125F,
          0x1287..0x1287,
          0x1289..0x1289,
          0x128E..0x128F,
          0x12AF..0x12AF,
          0x12B1..0x12B1,
          0x12B6..0x12B7,
          0x12BF..0x12BF,
          0x12C1..0x12C1,
          0x12C6..0x12C7,
          0x12CF..0x12CF,
          0x12D7..0x12D7,
          0x12EF..0x12EF,
          0x130F..0x130F,
          0x1311..0x1311,
          0x1316..0x1317,
          0x131F..0x131F,
          0x1347..0x1347,
          0x135B..0x1360,
          0x137D..0x139F,
          0x13F5..0x1400,
          0x1677..0x167F,
          0x169D..0x169F,
          0x16F1..0x16FF,
          0x170D..0x170D,
          0x1715..0x171F,
          0x1737..0x173F,
          0x1754..0x175F,
          0x176D..0x176D,
          0x1771..0x1771,
          0x1774..0x177F,
          0x17DD..0x17DF,
          0x17EA..0x17FF,
          0x180F..0x180F,
          0x181A..0x181F,
          0x1878..0x187F,
          0x18AA..0x1DFF,
          0x1E9C..0x1E9F,
          0x1EFA..0x1EFF,
          0x1F16..0x1F17,
          0x1F1E..0x1F1F,
          0x1F46..0x1F47,
          0x1F4E..0x1F4F,
          0x1F58..0x1F58,
          0x1F5A..0x1F5A,
          0x1F5C..0x1F5C,
          0x1F5E..0x1F5E,
          0x1F7E..0x1F7F,
          0x1FB5..0x1FB5,
          0x1FC5..0x1FC5,
          0x1FD4..0x1FD5,
          0x1FDC..0x1FDC,
          0x1FF0..0x1FF1,
          0x1FF5..0x1FF5,
          0x1FFF..0x1FFF,
          0x2053..0x2056,
          0x2058..0x205E,
          0x2064..0x2069,
          0x2072..0x2073,
          0x208F..0x209F,
          0x20B2..0x20CF,
          0x20EB..0x20FF,
          0x213B..0x213C,
          0x214C..0x2152,
          0x2184..0x218F,
          0x23CF..0x23FF,
          0x2427..0x243F,
          0x244B..0x245F,
          0x24FF..0x24FF,
          0x2614..0x2615,
          0x2618..0x2618,
          0x267E..0x267F,
          0x268A..0x2700,
          0x2705..0x2705,
          0x270A..0x270B,
          0x2728..0x2728,
          0x274C..0x274C,
          0x274E..0x274E,
          0x2753..0x2755,
          0x2757..0x2757,
          0x275F..0x2760,
          0x2795..0x2797,
          0x27B0..0x27B0,
          0x27BF..0x27CF,
          0x27EC..0x27EF,
          0x2B00..0x2E7F,
          0x2E9A..0x2E9A,
          0x2EF4..0x2EFF,
          0x2FD6..0x2FEF,
          0x2FFC..0x2FFF,
          0x3040..0x3040,
          0x3097..0x3098,
          0x3100..0x3104,
          0x312D..0x3130,
          0x318F..0x318F,
          0x31B8..0x31EF,
          0x321D..0x321F,
          0x3244..0x3250,
          0x327C..0x327E,
          0x32CC..0x32CF,
          0x32FF..0x32FF,
          0x3377..0x337A,
          0x33DE..0x33DF,
          0x33FF..0x33FF,
          0x4DB6..0x4DFF,
          0x9FA6..0x9FFF,
          0xA48D..0xA48F,
          0xA4C7..0xABFF,
          0xD7A4..0xD7FF,
          0xFA2E..0xFA2F,
          0xFA6B..0xFAFF,
          0xFB07..0xFB12,
          0xFB18..0xFB1C,
          0xFB37..0xFB37,
          0xFB3D..0xFB3D,
          0xFB3F..0xFB3F,
          0xFB42..0xFB42,
          0xFB45..0xFB45,
          0xFBB2..0xFBD2,
          0xFD40..0xFD4F,
          0xFD90..0xFD91,
          0xFDC8..0xFDCF,
          0xFDFD..0xFDFF,
          0xFE10..0xFE1F,
          0xFE24..0xFE2F,
          0xFE47..0xFE48,
          0xFE53..0xFE53,
          0xFE67..0xFE67,
          0xFE6C..0xFE6F,
          0xFE75..0xFE75,
          0xFEFD..0xFEFE,
          0xFF00..0xFF00,
          0xFFBF..0xFFC1,
          0xFFC8..0xFFC9,
          0xFFD0..0xFFD1,
          0xFFD8..0xFFD9,
          0xFFDD..0xFFDF,
          0xFFE7..0xFFE7,
          0xFFEF..0xFFF8,
          0x10000..0x102FF,
          0x1031F..0x1031F,
          0x10324..0x1032F,
          0x1034B..0x103FF,
          0x10426..0x10427,
          0x1044E..0x1CFFF,
          0x1D0F6..0x1D0FF,
          0x1D127..0x1D129,
          0x1D1DE..0x1D3FF,
          0x1D455..0x1D455,
          0x1D49D..0x1D49D,
          0x1D4A0..0x1D4A1,
          0x1D4A3..0x1D4A4,
          0x1D4A7..0x1D4A8,
          0x1D4AD..0x1D4AD,
          0x1D4BA..0x1D4BA,
          0x1D4BC..0x1D4BC,
          0x1D4C1..0x1D4C1,
          0x1D4C4..0x1D4C4,
          0x1D506..0x1D506,
          0x1D50B..0x1D50C,
          0x1D515..0x1D515,
          0x1D51D..0x1D51D,
          0x1D53A..0x1D53A,
          0x1D53F..0x1D53F,
          0x1D545..0x1D545,
          0x1D547..0x1D549,
          0x1D551..0x1D551,
          0x1D6A4..0x1D6A7,
          0x1D7CA..0x1D7CD,
          0x1D800..0x1FFFD,
          0x2A6D7..0x2F7FF,
          0x2FA1E..0x2FFFD,
          0x30000..0x3FFFD,
          0x40000..0x4FFFD,
          0x50000..0x5FFFD,
          0x60000..0x6FFFD,
          0x70000..0x7FFFD,
          0x80000..0x8FFFD,
          0x90000..0x9FFFD,
          0xA0000..0xAFFFD,
          0xB0000..0xBFFFD,
          0xC0000..0xCFFFD,
          0xD0000..0xDFFFD,
          0xE0000..0xE0000,
          0xE0002..0xE001F,
          0xE0080..0xEFFFD,
        ].freeze

        # Table B1 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        B1 = {
          0x00AD => [], # Map to nothing
          0x034F => [], # Map to nothing
          0x180B => [], # Map to nothing
          0x180C => [], # Map to nothing
          0x180D => [], # Map to nothing
          0x200B => [], # Map to nothing
          0x200C => [], # Map to nothing
          0x200D => [], # Map to nothing
          0x2060 => [], # Map to nothing
          0xFE00 => [], # Map to nothing
          0xFE01 => [], # Map to nothing
          0xFE02 => [], # Map to nothing
          0xFE03 => [], # Map to nothing
          0xFE04 => [], # Map to nothing
          0xFE05 => [], # Map to nothing
          0xFE06 => [], # Map to nothing
          0xFE07 => [], # Map to nothing
          0xFE08 => [], # Map to nothing
          0xFE09 => [], # Map to nothing
          0xFE0A => [], # Map to nothing
          0xFE0B => [], # Map to nothing
          0xFE0C => [], # Map to nothing
          0xFE0D => [], # Map to nothing
          0xFE0E => [], # Map to nothing
          0xFE0F => [], # Map to nothing
          0xFEFF => [], # Map to nothing
        }.freeze

        # Table B2 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        B2 = {
          0x0041 => [0x0061], # Case map
          0x0042 => [0x0062], # Case map
          0x0043 => [0x0063], # Case map
          0x0044 => [0x0064], # Case map
          0x0045 => [0x0065], # Case map
          0x0046 => [0x0066], # Case map
          0x0047 => [0x0067], # Case map
          0x0048 => [0x0068], # Case map
          0x0049 => [0x0069], # Case map
          0x004A => [0x006A], # Case map
          0x004B => [0x006B], # Case map
          0x004C => [0x006C], # Case map
          0x004D => [0x006D], # Case map
          0x004E => [0x006E], # Case map
          0x004F => [0x006F], # Case map
          0x0050 => [0x0070], # Case map
          0x0051 => [0x0071], # Case map
          0x0052 => [0x0072], # Case map
          0x0053 => [0x0073], # Case map
          0x0054 => [0x0074], # Case map
          0x0055 => [0x0075], # Case map
          0x0056 => [0x0076], # Case map
          0x0057 => [0x0077], # Case map
          0x0058 => [0x0078], # Case map
          0x0059 => [0x0079], # Case map
          0x005A => [0x007A], # Case map
          0x00B5 => [0x03BC], # Case map
          0x00C0 => [0x00E0], # Case map
          0x00C1 => [0x00E1], # Case map
          0x00C2 => [0x00E2], # Case map
          0x00C3 => [0x00E3], # Case map
          0x00C4 => [0x00E4], # Case map
          0x00C5 => [0x00E5], # Case map
          0x00C6 => [0x00E6], # Case map
          0x00C7 => [0x00E7], # Case map
          0x00C8 => [0x00E8], # Case map
          0x00C9 => [0x00E9], # Case map
          0x00CA => [0x00EA], # Case map
          0x00CB => [0x00EB], # Case map
          0x00CC => [0x00EC], # Case map
          0x00CD => [0x00ED], # Case map
          0x00CE => [0x00EE], # Case map
          0x00CF => [0x00EF], # Case map
          0x00D0 => [0x00F0], # Case map
          0x00D1 => [0x00F1], # Case map
          0x00D2 => [0x00F2], # Case map
          0x00D3 => [0x00F3], # Case map
          0x00D4 => [0x00F4], # Case map
          0x00D5 => [0x00F5], # Case map
          0x00D6 => [0x00F6], # Case map
          0x00D8 => [0x00F8], # Case map
          0x00D9 => [0x00F9], # Case map
          0x00DA => [0x00FA], # Case map
          0x00DB => [0x00FB], # Case map
          0x00DC => [0x00FC], # Case map
          0x00DD => [0x00FD], # Case map
          0x00DE => [0x00FE], # Case map
          0x00DF => [0x0073, 0x0073], # Case map
          0x0100 => [0x0101], # Case map
          0x0102 => [0x0103], # Case map
          0x0104 => [0x0105], # Case map
          0x0106 => [0x0107], # Case map
          0x0108 => [0x0109], # Case map
          0x010A => [0x010B], # Case map
          0x010C => [0x010D], # Case map
          0x010E => [0x010F], # Case map
          0x0110 => [0x0111], # Case map
          0x0112 => [0x0113], # Case map
          0x0114 => [0x0115], # Case map
          0x0116 => [0x0117], # Case map
          0x0118 => [0x0119], # Case map
          0x011A => [0x011B], # Case map
          0x011C => [0x011D], # Case map
          0x011E => [0x011F], # Case map
          0x0120 => [0x0121], # Case map
          0x0122 => [0x0123], # Case map
          0x0124 => [0x0125], # Case map
          0x0126 => [0x0127], # Case map
          0x0128 => [0x0129], # Case map
          0x012A => [0x012B], # Case map
          0x012C => [0x012D], # Case map
          0x012E => [0x012F], # Case map
          0x0130 => [0x0069, 0x0307], # Case map
          0x0132 => [0x0133], # Case map
          0x0134 => [0x0135], # Case map
          0x0136 => [0x0137], # Case map
          0x0139 => [0x013A], # Case map
          0x013B => [0x013C], # Case map
          0x013D => [0x013E], # Case map
          0x013F => [0x0140], # Case map
          0x0141 => [0x0142], # Case map
          0x0143 => [0x0144], # Case map
          0x0145 => [0x0146], # Case map
          0x0147 => [0x0148], # Case map
          0x0149 => [0x02BC, 0x006E], # Case map
          0x014A => [0x014B], # Case map
          0x014C => [0x014D], # Case map
          0x014E => [0x014F], # Case map
          0x0150 => [0x0151], # Case map
          0x0152 => [0x0153], # Case map
          0x0154 => [0x0155], # Case map
          0x0156 => [0x0157], # Case map
          0x0158 => [0x0159], # Case map
          0x015A => [0x015B], # Case map
          0x015C => [0x015D], # Case map
          0x015E => [0x015F], # Case map
          0x0160 => [0x0161], # Case map
          0x0162 => [0x0163], # Case map
          0x0164 => [0x0165], # Case map
          0x0166 => [0x0167], # Case map
          0x0168 => [0x0169], # Case map
          0x016A => [0x016B], # Case map
          0x016C => [0x016D], # Case map
          0x016E => [0x016F], # Case map
          0x0170 => [0x0171], # Case map
          0x0172 => [0x0173], # Case map
          0x0174 => [0x0175], # Case map
          0x0176 => [0x0177], # Case map
          0x0178 => [0x00FF], # Case map
          0x0179 => [0x017A], # Case map
          0x017B => [0x017C], # Case map
          0x017D => [0x017E], # Case map
          0x017F => [0x0073], # Case map
          0x0181 => [0x0253], # Case map
          0x0182 => [0x0183], # Case map
          0x0184 => [0x0185], # Case map
          0x0186 => [0x0254], # Case map
          0x0187 => [0x0188], # Case map
          0x0189 => [0x0256], # Case map
          0x018A => [0x0257], # Case map
          0x018B => [0x018C], # Case map
          0x018E => [0x01DD], # Case map
          0x018F => [0x0259], # Case map
          0x0190 => [0x025B], # Case map
          0x0191 => [0x0192], # Case map
          0x0193 => [0x0260], # Case map
          0x0194 => [0x0263], # Case map
          0x0196 => [0x0269], # Case map
          0x0197 => [0x0268], # Case map
          0x0198 => [0x0199], # Case map
          0x019C => [0x026F], # Case map
          0x019D => [0x0272], # Case map
          0x019F => [0x0275], # Case map
          0x01A0 => [0x01A1], # Case map
          0x01A2 => [0x01A3], # Case map
          0x01A4 => [0x01A5], # Case map
          0x01A6 => [0x0280], # Case map
          0x01A7 => [0x01A8], # Case map
          0x01A9 => [0x0283], # Case map
          0x01AC => [0x01AD], # Case map
          0x01AE => [0x0288], # Case map
          0x01AF => [0x01B0], # Case map
          0x01B1 => [0x028A], # Case map
          0x01B2 => [0x028B], # Case map
          0x01B3 => [0x01B4], # Case map
          0x01B5 => [0x01B6], # Case map
          0x01B7 => [0x0292], # Case map
          0x01B8 => [0x01B9], # Case map
          0x01BC => [0x01BD], # Case map
          0x01C4 => [0x01C6], # Case map
          0x01C5 => [0x01C6], # Case map
          0x01C7 => [0x01C9], # Case map
          0x01C8 => [0x01C9], # Case map
          0x01CA => [0x01CC], # Case map
          0x01CB => [0x01CC], # Case map
          0x01CD => [0x01CE], # Case map
          0x01CF => [0x01D0], # Case map
          0x01D1 => [0x01D2], # Case map
          0x01D3 => [0x01D4], # Case map
          0x01D5 => [0x01D6], # Case map
          0x01D7 => [0x01D8], # Case map
          0x01D9 => [0x01DA], # Case map
          0x01DB => [0x01DC], # Case map
          0x01DE => [0x01DF], # Case map
          0x01E0 => [0x01E1], # Case map
          0x01E2 => [0x01E3], # Case map
          0x01E4 => [0x01E5], # Case map
          0x01E6 => [0x01E7], # Case map
          0x01E8 => [0x01E9], # Case map
          0x01EA => [0x01EB], # Case map
          0x01EC => [0x01ED], # Case map
          0x01EE => [0x01EF], # Case map
          0x01F0 => [0x006A, 0x030C], # Case map
          0x01F1 => [0x01F3], # Case map
          0x01F2 => [0x01F3], # Case map
          0x01F4 => [0x01F5], # Case map
          0x01F6 => [0x0195], # Case map
          0x01F7 => [0x01BF], # Case map
          0x01F8 => [0x01F9], # Case map
          0x01FA => [0x01FB], # Case map
          0x01FC => [0x01FD], # Case map
          0x01FE => [0x01FF], # Case map
          0x0200 => [0x0201], # Case map
          0x0202 => [0x0203], # Case map
          0x0204 => [0x0205], # Case map
          0x0206 => [0x0207], # Case map
          0x0208 => [0x0209], # Case map
          0x020A => [0x020B], # Case map
          0x020C => [0x020D], # Case map
          0x020E => [0x020F], # Case map
          0x0210 => [0x0211], # Case map
          0x0212 => [0x0213], # Case map
          0x0214 => [0x0215], # Case map
          0x0216 => [0x0217], # Case map
          0x0218 => [0x0219], # Case map
          0x021A => [0x021B], # Case map
          0x021C => [0x021D], # Case map
          0x021E => [0x021F], # Case map
          0x0220 => [0x019E], # Case map
          0x0222 => [0x0223], # Case map
          0x0224 => [0x0225], # Case map
          0x0226 => [0x0227], # Case map
          0x0228 => [0x0229], # Case map
          0x022A => [0x022B], # Case map
          0x022C => [0x022D], # Case map
          0x022E => [0x022F], # Case map
          0x0230 => [0x0231], # Case map
          0x0232 => [0x0233], # Case map
          0x0345 => [0x03B9], # Case map
          0x037A => [0x0020, 0x03B9], # Additional folding
          0x0386 => [0x03AC], # Case map
          0x0388 => [0x03AD], # Case map
          0x0389 => [0x03AE], # Case map
          0x038A => [0x03AF], # Case map
          0x038C => [0x03CC], # Case map
          0x038E => [0x03CD], # Case map
          0x038F => [0x03CE], # Case map
          0x0390 => [0x03B9, 0x0308, 0x0301], # Case map
          0x0391 => [0x03B1], # Case map
          0x0392 => [0x03B2], # Case map
          0x0393 => [0x03B3], # Case map
          0x0394 => [0x03B4], # Case map
          0x0395 => [0x03B5], # Case map
          0x0396 => [0x03B6], # Case map
          0x0397 => [0x03B7], # Case map
          0x0398 => [0x03B8], # Case map
          0x0399 => [0x03B9], # Case map
          0x039A => [0x03BA], # Case map
          0x039B => [0x03BB], # Case map
          0x039C => [0x03BC], # Case map
          0x039D => [0x03BD], # Case map
          0x039E => [0x03BE], # Case map
          0x039F => [0x03BF], # Case map
          0x03A0 => [0x03C0], # Case map
          0x03A1 => [0x03C1], # Case map
          0x03A3 => [0x03C3], # Case map
          0x03A4 => [0x03C4], # Case map
          0x03A5 => [0x03C5], # Case map
          0x03A6 => [0x03C6], # Case map
          0x03A7 => [0x03C7], # Case map
          0x03A8 => [0x03C8], # Case map
          0x03A9 => [0x03C9], # Case map
          0x03AA => [0x03CA], # Case map
          0x03AB => [0x03CB], # Case map
          0x03B0 => [0x03C5, 0x0308, 0x0301], # Case map
          0x03C2 => [0x03C3], # Case map
          0x03D0 => [0x03B2], # Case map
          0x03D1 => [0x03B8], # Case map
          0x03D2 => [0x03C5], # Additional folding
          0x03D3 => [0x03CD], # Additional folding
          0x03D4 => [0x03CB], # Additional folding
          0x03D5 => [0x03C6], # Case map
          0x03D6 => [0x03C0], # Case map
          0x03D8 => [0x03D9], # Case map
          0x03DA => [0x03DB], # Case map
          0x03DC => [0x03DD], # Case map
          0x03DE => [0x03DF], # Case map
          0x03E0 => [0x03E1], # Case map
          0x03E2 => [0x03E3], # Case map
          0x03E4 => [0x03E5], # Case map
          0x03E6 => [0x03E7], # Case map
          0x03E8 => [0x03E9], # Case map
          0x03EA => [0x03EB], # Case map
          0x03EC => [0x03ED], # Case map
          0x03EE => [0x03EF], # Case map
          0x03F0 => [0x03BA], # Case map
          0x03F1 => [0x03C1], # Case map
          0x03F2 => [0x03C3], # Case map
          0x03F4 => [0x03B8], # Case map
          0x03F5 => [0x03B5], # Case map
          0x0400 => [0x0450], # Case map
          0x0401 => [0x0451], # Case map
          0x0402 => [0x0452], # Case map
          0x0403 => [0x0453], # Case map
          0x0404 => [0x0454], # Case map
          0x0405 => [0x0455], # Case map
          0x0406 => [0x0456], # Case map
          0x0407 => [0x0457], # Case map
          0x0408 => [0x0458], # Case map
          0x0409 => [0x0459], # Case map
          0x040A => [0x045A], # Case map
          0x040B => [0x045B], # Case map
          0x040C => [0x045C], # Case map
          0x040D => [0x045D], # Case map
          0x040E => [0x045E], # Case map
          0x040F => [0x045F], # Case map
          0x0410 => [0x0430], # Case map
          0x0411 => [0x0431], # Case map
          0x0412 => [0x0432], # Case map
          0x0413 => [0x0433], # Case map
          0x0414 => [0x0434], # Case map
          0x0415 => [0x0435], # Case map
          0x0416 => [0x0436], # Case map
          0x0417 => [0x0437], # Case map
          0x0418 => [0x0438], # Case map
          0x0419 => [0x0439], # Case map
          0x041A => [0x043A], # Case map
          0x041B => [0x043B], # Case map
          0x041C => [0x043C], # Case map
          0x041D => [0x043D], # Case map
          0x041E => [0x043E], # Case map
          0x041F => [0x043F], # Case map
          0x0420 => [0x0440], # Case map
          0x0421 => [0x0441], # Case map
          0x0422 => [0x0442], # Case map
          0x0423 => [0x0443], # Case map
          0x0424 => [0x0444], # Case map
          0x0425 => [0x0445], # Case map
          0x0426 => [0x0446], # Case map
          0x0427 => [0x0447], # Case map
          0x0428 => [0x0448], # Case map
          0x0429 => [0x0449], # Case map
          0x042A => [0x044A], # Case map
          0x042B => [0x044B], # Case map
          0x042C => [0x044C], # Case map
          0x042D => [0x044D], # Case map
          0x042E => [0x044E], # Case map
          0x042F => [0x044F], # Case map
          0x0460 => [0x0461], # Case map
          0x0462 => [0x0463], # Case map
          0x0464 => [0x0465], # Case map
          0x0466 => [0x0467], # Case map
          0x0468 => [0x0469], # Case map
          0x046A => [0x046B], # Case map
          0x046C => [0x046D], # Case map
          0x046E => [0x046F], # Case map
          0x0470 => [0x0471], # Case map
          0x0472 => [0x0473], # Case map
          0x0474 => [0x0475], # Case map
          0x0476 => [0x0477], # Case map
          0x0478 => [0x0479], # Case map
          0x047A => [0x047B], # Case map
          0x047C => [0x047D], # Case map
          0x047E => [0x047F], # Case map
          0x0480 => [0x0481], # Case map
          0x048A => [0x048B], # Case map
          0x048C => [0x048D], # Case map
          0x048E => [0x048F], # Case map
          0x0490 => [0x0491], # Case map
          0x0492 => [0x0493], # Case map
          0x0494 => [0x0495], # Case map
          0x0496 => [0x0497], # Case map
          0x0498 => [0x0499], # Case map
          0x049A => [0x049B], # Case map
          0x049C => [0x049D], # Case map
          0x049E => [0x049F], # Case map
          0x04A0 => [0x04A1], # Case map
          0x04A2 => [0x04A3], # Case map
          0x04A4 => [0x04A5], # Case map
          0x04A6 => [0x04A7], # Case map
          0x04A8 => [0x04A9], # Case map
          0x04AA => [0x04AB], # Case map
          0x04AC => [0x04AD], # Case map
          0x04AE => [0x04AF], # Case map
          0x04B0 => [0x04B1], # Case map
          0x04B2 => [0x04B3], # Case map
          0x04B4 => [0x04B5], # Case map
          0x04B6 => [0x04B7], # Case map
          0x04B8 => [0x04B9], # Case map
          0x04BA => [0x04BB], # Case map
          0x04BC => [0x04BD], # Case map
          0x04BE => [0x04BF], # Case map
          0x04C1 => [0x04C2], # Case map
          0x04C3 => [0x04C4], # Case map
          0x04C5 => [0x04C6], # Case map
          0x04C7 => [0x04C8], # Case map
          0x04C9 => [0x04CA], # Case map
          0x04CB => [0x04CC], # Case map
          0x04CD => [0x04CE], # Case map
          0x04D0 => [0x04D1], # Case map
          0x04D2 => [0x04D3], # Case map
          0x04D4 => [0x04D5], # Case map
          0x04D6 => [0x04D7], # Case map
          0x04D8 => [0x04D9], # Case map
          0x04DA => [0x04DB], # Case map
          0x04DC => [0x04DD], # Case map
          0x04DE => [0x04DF], # Case map
          0x04E0 => [0x04E1], # Case map
          0x04E2 => [0x04E3], # Case map
          0x04E4 => [0x04E5], # Case map
          0x04E6 => [0x04E7], # Case map
          0x04E8 => [0x04E9], # Case map
          0x04EA => [0x04EB], # Case map
          0x04EC => [0x04ED], # Case map
          0x04EE => [0x04EF], # Case map
          0x04F0 => [0x04F1], # Case map
          0x04F2 => [0x04F3], # Case map
          0x04F4 => [0x04F5], # Case map
          0x04F8 => [0x04F9], # Case map
          0x0500 => [0x0501], # Case map
          0x0502 => [0x0503], # Case map
          0x0504 => [0x0505], # Case map
          0x0506 => [0x0507], # Case map
          0x0508 => [0x0509], # Case map
          0x050A => [0x050B], # Case map
          0x050C => [0x050D], # Case map
          0x050E => [0x050F], # Case map
          0x0531 => [0x0561], # Case map
          0x0532 => [0x0562], # Case map
          0x0533 => [0x0563], # Case map
          0x0534 => [0x0564], # Case map
          0x0535 => [0x0565], # Case map
          0x0536 => [0x0566], # Case map
          0x0537 => [0x0567], # Case map
          0x0538 => [0x0568], # Case map
          0x0539 => [0x0569], # Case map
          0x053A => [0x056A], # Case map
          0x053B => [0x056B], # Case map
          0x053C => [0x056C], # Case map
          0x053D => [0x056D], # Case map
          0x053E => [0x056E], # Case map
          0x053F => [0x056F], # Case map
          0x0540 => [0x0570], # Case map
          0x0541 => [0x0571], # Case map
          0x0542 => [0x0572], # Case map
          0x0543 => [0x0573], # Case map
          0x0544 => [0x0574], # Case map
          0x0545 => [0x0575], # Case map
          0x0546 => [0x0576], # Case map
          0x0547 => [0x0577], # Case map
          0x0548 => [0x0578], # Case map
          0x0549 => [0x0579], # Case map
          0x054A => [0x057A], # Case map
          0x054B => [0x057B], # Case map
          0x054C => [0x057C], # Case map
          0x054D => [0x057D], # Case map
          0x054E => [0x057E], # Case map
          0x054F => [0x057F], # Case map
          0x0550 => [0x0580], # Case map
          0x0551 => [0x0581], # Case map
          0x0552 => [0x0582], # Case map
          0x0553 => [0x0583], # Case map
          0x0554 => [0x0584], # Case map
          0x0555 => [0x0585], # Case map
          0x0556 => [0x0586], # Case map
          0x0587 => [0x0565, 0x0582], # Case map
          0x1E00 => [0x1E01], # Case map
          0x1E02 => [0x1E03], # Case map
          0x1E04 => [0x1E05], # Case map
          0x1E06 => [0x1E07], # Case map
          0x1E08 => [0x1E09], # Case map
          0x1E0A => [0x1E0B], # Case map
          0x1E0C => [0x1E0D], # Case map
          0x1E0E => [0x1E0F], # Case map
          0x1E10 => [0x1E11], # Case map
          0x1E12 => [0x1E13], # Case map
          0x1E14 => [0x1E15], # Case map
          0x1E16 => [0x1E17], # Case map
          0x1E18 => [0x1E19], # Case map
          0x1E1A => [0x1E1B], # Case map
          0x1E1C => [0x1E1D], # Case map
          0x1E1E => [0x1E1F], # Case map
          0x1E20 => [0x1E21], # Case map
          0x1E22 => [0x1E23], # Case map
          0x1E24 => [0x1E25], # Case map
          0x1E26 => [0x1E27], # Case map
          0x1E28 => [0x1E29], # Case map
          0x1E2A => [0x1E2B], # Case map
          0x1E2C => [0x1E2D], # Case map
          0x1E2E => [0x1E2F], # Case map
          0x1E30 => [0x1E31], # Case map
          0x1E32 => [0x1E33], # Case map
          0x1E34 => [0x1E35], # Case map
          0x1E36 => [0x1E37], # Case map
          0x1E38 => [0x1E39], # Case map
          0x1E3A => [0x1E3B], # Case map
          0x1E3C => [0x1E3D], # Case map
          0x1E3E => [0x1E3F], # Case map
          0x1E40 => [0x1E41], # Case map
          0x1E42 => [0x1E43], # Case map
          0x1E44 => [0x1E45], # Case map
          0x1E46 => [0x1E47], # Case map
          0x1E48 => [0x1E49], # Case map
          0x1E4A => [0x1E4B], # Case map
          0x1E4C => [0x1E4D], # Case map
          0x1E4E => [0x1E4F], # Case map
          0x1E50 => [0x1E51], # Case map
          0x1E52 => [0x1E53], # Case map
          0x1E54 => [0x1E55], # Case map
          0x1E56 => [0x1E57], # Case map
          0x1E58 => [0x1E59], # Case map
          0x1E5A => [0x1E5B], # Case map
          0x1E5C => [0x1E5D], # Case map
          0x1E5E => [0x1E5F], # Case map
          0x1E60 => [0x1E61], # Case map
          0x1E62 => [0x1E63], # Case map
          0x1E64 => [0x1E65], # Case map
          0x1E66 => [0x1E67], # Case map
          0x1E68 => [0x1E69], # Case map
          0x1E6A => [0x1E6B], # Case map
          0x1E6C => [0x1E6D], # Case map
          0x1E6E => [0x1E6F], # Case map
          0x1E70 => [0x1E71], # Case map
          0x1E72 => [0x1E73], # Case map
          0x1E74 => [0x1E75], # Case map
          0x1E76 => [0x1E77], # Case map
          0x1E78 => [0x1E79], # Case map
          0x1E7A => [0x1E7B], # Case map
          0x1E7C => [0x1E7D], # Case map
          0x1E7E => [0x1E7F], # Case map
          0x1E80 => [0x1E81], # Case map
          0x1E82 => [0x1E83], # Case map
          0x1E84 => [0x1E85], # Case map
          0x1E86 => [0x1E87], # Case map
          0x1E88 => [0x1E89], # Case map
          0x1E8A => [0x1E8B], # Case map
          0x1E8C => [0x1E8D], # Case map
          0x1E8E => [0x1E8F], # Case map
          0x1E90 => [0x1E91], # Case map
          0x1E92 => [0x1E93], # Case map
          0x1E94 => [0x1E95], # Case map
          0x1E96 => [0x0068, 0x0331], # Case map
          0x1E97 => [0x0074, 0x0308], # Case map
          0x1E98 => [0x0077, 0x030A], # Case map
          0x1E99 => [0x0079, 0x030A], # Case map
          0x1E9A => [0x0061, 0x02BE], # Case map
          0x1E9B => [0x1E61], # Case map
          0x1EA0 => [0x1EA1], # Case map
          0x1EA2 => [0x1EA3], # Case map
          0x1EA4 => [0x1EA5], # Case map
          0x1EA6 => [0x1EA7], # Case map
          0x1EA8 => [0x1EA9], # Case map
          0x1EAA => [0x1EAB], # Case map
          0x1EAC => [0x1EAD], # Case map
          0x1EAE => [0x1EAF], # Case map
          0x1EB0 => [0x1EB1], # Case map
          0x1EB2 => [0x1EB3], # Case map
          0x1EB4 => [0x1EB5], # Case map
          0x1EB6 => [0x1EB7], # Case map
          0x1EB8 => [0x1EB9], # Case map
          0x1EBA => [0x1EBB], # Case map
          0x1EBC => [0x1EBD], # Case map
          0x1EBE => [0x1EBF], # Case map
          0x1EC0 => [0x1EC1], # Case map
          0x1EC2 => [0x1EC3], # Case map
          0x1EC4 => [0x1EC5], # Case map
          0x1EC6 => [0x1EC7], # Case map
          0x1EC8 => [0x1EC9], # Case map
          0x1ECA => [0x1ECB], # Case map
          0x1ECC => [0x1ECD], # Case map
          0x1ECE => [0x1ECF], # Case map
          0x1ED0 => [0x1ED1], # Case map
          0x1ED2 => [0x1ED3], # Case map
          0x1ED4 => [0x1ED5], # Case map
          0x1ED6 => [0x1ED7], # Case map
          0x1ED8 => [0x1ED9], # Case map
          0x1EDA => [0x1EDB], # Case map
          0x1EDC => [0x1EDD], # Case map
          0x1EDE => [0x1EDF], # Case map
          0x1EE0 => [0x1EE1], # Case map
          0x1EE2 => [0x1EE3], # Case map
          0x1EE4 => [0x1EE5], # Case map
          0x1EE6 => [0x1EE7], # Case map
          0x1EE8 => [0x1EE9], # Case map
          0x1EEA => [0x1EEB], # Case map
          0x1EEC => [0x1EED], # Case map
          0x1EEE => [0x1EEF], # Case map
          0x1EF0 => [0x1EF1], # Case map
          0x1EF2 => [0x1EF3], # Case map
          0x1EF4 => [0x1EF5], # Case map
          0x1EF6 => [0x1EF7], # Case map
          0x1EF8 => [0x1EF9], # Case map
          0x1F08 => [0x1F00], # Case map
          0x1F09 => [0x1F01], # Case map
          0x1F0A => [0x1F02], # Case map
          0x1F0B => [0x1F03], # Case map
          0x1F0C => [0x1F04], # Case map
          0x1F0D => [0x1F05], # Case map
          0x1F0E => [0x1F06], # Case map
          0x1F0F => [0x1F07], # Case map
          0x1F18 => [0x1F10], # Case map
          0x1F19 => [0x1F11], # Case map
          0x1F1A => [0x1F12], # Case map
          0x1F1B => [0x1F13], # Case map
          0x1F1C => [0x1F14], # Case map
          0x1F1D => [0x1F15], # Case map
          0x1F28 => [0x1F20], # Case map
          0x1F29 => [0x1F21], # Case map
          0x1F2A => [0x1F22], # Case map
          0x1F2B => [0x1F23], # Case map
          0x1F2C => [0x1F24], # Case map
          0x1F2D => [0x1F25], # Case map
          0x1F2E => [0x1F26], # Case map
          0x1F2F => [0x1F27], # Case map
          0x1F38 => [0x1F30], # Case map
          0x1F39 => [0x1F31], # Case map
          0x1F3A => [0x1F32], # Case map
          0x1F3B => [0x1F33], # Case map
          0x1F3C => [0x1F34], # Case map
          0x1F3D => [0x1F35], # Case map
          0x1F3E => [0x1F36], # Case map
          0x1F3F => [0x1F37], # Case map
          0x1F48 => [0x1F40], # Case map
          0x1F49 => [0x1F41], # Case map
          0x1F4A => [0x1F42], # Case map
          0x1F4B => [0x1F43], # Case map
          0x1F4C => [0x1F44], # Case map
          0x1F4D => [0x1F45], # Case map
          0x1F50 => [0x03C5, 0x0313], # Case map
          0x1F52 => [0x03C5, 0x0313, 0x0300], # Case map
          0x1F54 => [0x03C5, 0x0313, 0x0301], # Case map
          0x1F56 => [0x03C5, 0x0313, 0x0342], # Case map
          0x1F59 => [0x1F51], # Case map
          0x1F5B => [0x1F53], # Case map
          0x1F5D => [0x1F55], # Case map
          0x1F5F => [0x1F57], # Case map
          0x1F68 => [0x1F60], # Case map
          0x1F69 => [0x1F61], # Case map
          0x1F6A => [0x1F62], # Case map
          0x1F6B => [0x1F63], # Case map
          0x1F6C => [0x1F64], # Case map
          0x1F6D => [0x1F65], # Case map
          0x1F6E => [0x1F66], # Case map
          0x1F6F => [0x1F67], # Case map
          0x1F80 => [0x1F00, 0x03B9], # Case map
          0x1F81 => [0x1F01, 0x03B9], # Case map
          0x1F82 => [0x1F02, 0x03B9], # Case map
          0x1F83 => [0x1F03, 0x03B9], # Case map
          0x1F84 => [0x1F04, 0x03B9], # Case map
          0x1F85 => [0x1F05, 0x03B9], # Case map
          0x1F86 => [0x1F06, 0x03B9], # Case map
          0x1F87 => [0x1F07, 0x03B9], # Case map
          0x1F88 => [0x1F00, 0x03B9], # Case map
          0x1F89 => [0x1F01, 0x03B9], # Case map
          0x1F8A => [0x1F02, 0x03B9], # Case map
          0x1F8B => [0x1F03, 0x03B9], # Case map
          0x1F8C => [0x1F04, 0x03B9], # Case map
          0x1F8D => [0x1F05, 0x03B9], # Case map
          0x1F8E => [0x1F06, 0x03B9], # Case map
          0x1F8F => [0x1F07, 0x03B9], # Case map
          0x1F90 => [0x1F20, 0x03B9], # Case map
          0x1F91 => [0x1F21, 0x03B9], # Case map
          0x1F92 => [0x1F22, 0x03B9], # Case map
          0x1F93 => [0x1F23, 0x03B9], # Case map
          0x1F94 => [0x1F24, 0x03B9], # Case map
          0x1F95 => [0x1F25, 0x03B9], # Case map
          0x1F96 => [0x1F26, 0x03B9], # Case map
          0x1F97 => [0x1F27, 0x03B9], # Case map
          0x1F98 => [0x1F20, 0x03B9], # Case map
          0x1F99 => [0x1F21, 0x03B9], # Case map
          0x1F9A => [0x1F22, 0x03B9], # Case map
          0x1F9B => [0x1F23, 0x03B9], # Case map
          0x1F9C => [0x1F24, 0x03B9], # Case map
          0x1F9D => [0x1F25, 0x03B9], # Case map
          0x1F9E => [0x1F26, 0x03B9], # Case map
          0x1F9F => [0x1F27, 0x03B9], # Case map
          0x1FA0 => [0x1F60, 0x03B9], # Case map
          0x1FA1 => [0x1F61, 0x03B9], # Case map
          0x1FA2 => [0x1F62, 0x03B9], # Case map
          0x1FA3 => [0x1F63, 0x03B9], # Case map
          0x1FA4 => [0x1F64, 0x03B9], # Case map
          0x1FA5 => [0x1F65, 0x03B9], # Case map
          0x1FA6 => [0x1F66, 0x03B9], # Case map
          0x1FA7 => [0x1F67, 0x03B9], # Case map
          0x1FA8 => [0x1F60, 0x03B9], # Case map
          0x1FA9 => [0x1F61, 0x03B9], # Case map
          0x1FAA => [0x1F62, 0x03B9], # Case map
          0x1FAB => [0x1F63, 0x03B9], # Case map
          0x1FAC => [0x1F64, 0x03B9], # Case map
          0x1FAD => [0x1F65, 0x03B9], # Case map
          0x1FAE => [0x1F66, 0x03B9], # Case map
          0x1FAF => [0x1F67, 0x03B9], # Case map
          0x1FB2 => [0x1F70, 0x03B9], # Case map
          0x1FB3 => [0x03B1, 0x03B9], # Case map
          0x1FB4 => [0x03AC, 0x03B9], # Case map
          0x1FB6 => [0x03B1, 0x0342], # Case map
          0x1FB7 => [0x03B1, 0x0342, 0x03B9], # Case map
          0x1FB8 => [0x1FB0], # Case map
          0x1FB9 => [0x1FB1], # Case map
          0x1FBA => [0x1F70], # Case map
          0x1FBB => [0x1F71], # Case map
          0x1FBC => [0x03B1, 0x03B9], # Case map
          0x1FBE => [0x03B9], # Case map
          0x1FC2 => [0x1F74, 0x03B9], # Case map
          0x1FC3 => [0x03B7, 0x03B9], # Case map
          0x1FC4 => [0x03AE, 0x03B9], # Case map
          0x1FC6 => [0x03B7, 0x0342], # Case map
          0x1FC7 => [0x03B7, 0x0342, 0x03B9], # Case map
          0x1FC8 => [0x1F72], # Case map
          0x1FC9 => [0x1F73], # Case map
          0x1FCA => [0x1F74], # Case map
          0x1FCB => [0x1F75], # Case map
          0x1FCC => [0x03B7, 0x03B9], # Case map
          0x1FD2 => [0x03B9, 0x0308, 0x0300], # Case map
          0x1FD3 => [0x03B9, 0x0308, 0x0301], # Case map
          0x1FD6 => [0x03B9, 0x0342], # Case map
          0x1FD7 => [0x03B9, 0x0308, 0x0342], # Case map
          0x1FD8 => [0x1FD0], # Case map
          0x1FD9 => [0x1FD1], # Case map
          0x1FDA => [0x1F76], # Case map
          0x1FDB => [0x1F77], # Case map
          0x1FE2 => [0x03C5, 0x0308, 0x0300], # Case map
          0x1FE3 => [0x03C5, 0x0308, 0x0301], # Case map
          0x1FE4 => [0x03C1, 0x0313], # Case map
          0x1FE6 => [0x03C5, 0x0342], # Case map
          0x1FE7 => [0x03C5, 0x0308, 0x0342], # Case map
          0x1FE8 => [0x1FE0], # Case map
          0x1FE9 => [0x1FE1], # Case map
          0x1FEA => [0x1F7A], # Case map
          0x1FEB => [0x1F7B], # Case map
          0x1FEC => [0x1FE5], # Case map
          0x1FF2 => [0x1F7C, 0x03B9], # Case map
          0x1FF3 => [0x03C9, 0x03B9], # Case map
          0x1FF4 => [0x03CE, 0x03B9], # Case map
          0x1FF6 => [0x03C9, 0x0342], # Case map
          0x1FF7 => [0x03C9, 0x0342, 0x03B9], # Case map
          0x1FF8 => [0x1F78], # Case map
          0x1FF9 => [0x1F79], # Case map
          0x1FFA => [0x1F7C], # Case map
          0x1FFB => [0x1F7D], # Case map
          0x1FFC => [0x03C9, 0x03B9], # Case map
          0x20A8 => [0x0072, 0x0073], # Additional folding
          0x2102 => [0x0063], # Additional folding
          0x2103 => [0x00B0, 0x0063], # Additional folding
          0x2107 => [0x025B], # Additional folding
          0x2109 => [0x00B0, 0x0066], # Additional folding
          0x210B => [0x0068], # Additional folding
          0x210C => [0x0068], # Additional folding
          0x210D => [0x0068], # Additional folding
          0x2110 => [0x0069], # Additional folding
          0x2111 => [0x0069], # Additional folding
          0x2112 => [0x006C], # Additional folding
          0x2115 => [0x006E], # Additional folding
          0x2116 => [0x006E, 0x006F], # Additional folding
          0x2119 => [0x0070], # Additional folding
          0x211A => [0x0071], # Additional folding
          0x211B => [0x0072], # Additional folding
          0x211C => [0x0072], # Additional folding
          0x211D => [0x0072], # Additional folding
          0x2120 => [0x0073, 0x006D], # Additional folding
          0x2121 => [0x0074, 0x0065, 0x006C], # Additional folding
          0x2122 => [0x0074, 0x006D], # Additional folding
          0x2124 => [0x007A], # Additional folding
          0x2126 => [0x03C9], # Case map
          0x2128 => [0x007A], # Additional folding
          0x212A => [0x006B], # Case map
          0x212B => [0x00E5], # Case map
          0x212C => [0x0062], # Additional folding
          0x212D => [0x0063], # Additional folding
          0x2130 => [0x0065], # Additional folding
          0x2131 => [0x0066], # Additional folding
          0x2133 => [0x006D], # Additional folding
          0x213E => [0x03B3], # Additional folding
          0x213F => [0x03C0], # Additional folding
          0x2145 => [0x0064], # Additional folding
          0x2160 => [0x2170], # Case map
          0x2161 => [0x2171], # Case map
          0x2162 => [0x2172], # Case map
          0x2163 => [0x2173], # Case map
          0x2164 => [0x2174], # Case map
          0x2165 => [0x2175], # Case map
          0x2166 => [0x2176], # Case map
          0x2167 => [0x2177], # Case map
          0x2168 => [0x2178], # Case map
          0x2169 => [0x2179], # Case map
          0x216A => [0x217A], # Case map
          0x216B => [0x217B], # Case map
          0x216C => [0x217C], # Case map
          0x216D => [0x217D], # Case map
          0x216E => [0x217E], # Case map
          0x216F => [0x217F], # Case map
          0x24B6 => [0x24D0], # Case map
          0x24B7 => [0x24D1], # Case map
          0x24B8 => [0x24D2], # Case map
          0x24B9 => [0x24D3], # Case map
          0x24BA => [0x24D4], # Case map
          0x24BB => [0x24D5], # Case map
          0x24BC => [0x24D6], # Case map
          0x24BD => [0x24D7], # Case map
          0x24BE => [0x24D8], # Case map
          0x24BF => [0x24D9], # Case map
          0x24C0 => [0x24DA], # Case map
          0x24C1 => [0x24DB], # Case map
          0x24C2 => [0x24DC], # Case map
          0x24C3 => [0x24DD], # Case map
          0x24C4 => [0x24DE], # Case map
          0x24C5 => [0x24DF], # Case map
          0x24C6 => [0x24E0], # Case map
          0x24C7 => [0x24E1], # Case map
          0x24C8 => [0x24E2], # Case map
          0x24C9 => [0x24E3], # Case map
          0x24CA => [0x24E4], # Case map
          0x24CB => [0x24E5], # Case map
          0x24CC => [0x24E6], # Case map
          0x24CD => [0x24E7], # Case map
          0x24CE => [0x24E8], # Case map
          0x24CF => [0x24E9], # Case map
          0x3371 => [0x0068, 0x0070, 0x0061], # Additional folding
          0x3373 => [0x0061, 0x0075], # Additional folding
          0x3375 => [0x006F, 0x0076], # Additional folding
          0x3380 => [0x0070, 0x0061], # Additional folding
          0x3381 => [0x006E, 0x0061], # Additional folding
          0x3382 => [0x03BC, 0x0061], # Additional folding
          0x3383 => [0x006D, 0x0061], # Additional folding
          0x3384 => [0x006B, 0x0061], # Additional folding
          0x3385 => [0x006B, 0x0062], # Additional folding
          0x3386 => [0x006D, 0x0062], # Additional folding
          0x3387 => [0x0067, 0x0062], # Additional folding
          0x338A => [0x0070, 0x0066], # Additional folding
          0x338B => [0x006E, 0x0066], # Additional folding
          0x338C => [0x03BC, 0x0066], # Additional folding
          0x3390 => [0x0068, 0x007A], # Additional folding
          0x3391 => [0x006B, 0x0068, 0x007A], # Additional folding
          0x3392 => [0x006D, 0x0068, 0x007A], # Additional folding
          0x3393 => [0x0067, 0x0068, 0x007A], # Additional folding
          0x3394 => [0x0074, 0x0068, 0x007A], # Additional folding
          0x33A9 => [0x0070, 0x0061], # Additional folding
          0x33AA => [0x006B, 0x0070, 0x0061], # Additional folding
          0x33AB => [0x006D, 0x0070, 0x0061], # Additional folding
          0x33AC => [0x0067, 0x0070, 0x0061], # Additional folding
          0x33B4 => [0x0070, 0x0076], # Additional folding
          0x33B5 => [0x006E, 0x0076], # Additional folding
          0x33B6 => [0x03BC, 0x0076], # Additional folding
          0x33B7 => [0x006D, 0x0076], # Additional folding
          0x33B8 => [0x006B, 0x0076], # Additional folding
          0x33B9 => [0x006D, 0x0076], # Additional folding
          0x33BA => [0x0070, 0x0077], # Additional folding
          0x33BB => [0x006E, 0x0077], # Additional folding
          0x33BC => [0x03BC, 0x0077], # Additional folding
          0x33BD => [0x006D, 0x0077], # Additional folding
          0x33BE => [0x006B, 0x0077], # Additional folding
          0x33BF => [0x006D, 0x0077], # Additional folding
          0x33C0 => [0x006B, 0x03C9], # Additional folding
          0x33C1 => [0x006D, 0x03C9], # Additional folding
          0x33C3 => [0x0062, 0x0071], # Additional folding
          0x33C6 => [0x0063, 0x2215, 0x006B, 0x0067], # Additional folding
          0x33C7 => [0x0063, 0x006F, 0x002E], # Additional folding
          0x33C8 => [0x0064, 0x0062], # Additional folding
          0x33C9 => [0x0067, 0x0079], # Additional folding
          0x33CB => [0x0068, 0x0070], # Additional folding
          0x33CD => [0x006B, 0x006B], # Additional folding
          0x33CE => [0x006B, 0x006D], # Additional folding
          0x33D7 => [0x0070, 0x0068], # Additional folding
          0x33D9 => [0x0070, 0x0070, 0x006D], # Additional folding
          0x33DA => [0x0070, 0x0072], # Additional folding
          0x33DC => [0x0073, 0x0076], # Additional folding
          0x33DD => [0x0077, 0x0062], # Additional folding
          0xFB00 => [0x0066, 0x0066], # Case map
          0xFB01 => [0x0066, 0x0069], # Case map
          0xFB02 => [0x0066, 0x006C], # Case map
          0xFB03 => [0x0066, 0x0066, 0x0069], # Case map
          0xFB04 => [0x0066, 0x0066, 0x006C], # Case map
          0xFB05 => [0x0073, 0x0074], # Case map
          0xFB06 => [0x0073, 0x0074], # Case map
          0xFB13 => [0x0574, 0x0576], # Case map
          0xFB14 => [0x0574, 0x0565], # Case map
          0xFB15 => [0x0574, 0x056B], # Case map
          0xFB16 => [0x057E, 0x0576], # Case map
          0xFB17 => [0x0574, 0x056D], # Case map
          0xFF21 => [0xFF41], # Case map
          0xFF22 => [0xFF42], # Case map
          0xFF23 => [0xFF43], # Case map
          0xFF24 => [0xFF44], # Case map
          0xFF25 => [0xFF45], # Case map
          0xFF26 => [0xFF46], # Case map
          0xFF27 => [0xFF47], # Case map
          0xFF28 => [0xFF48], # Case map
          0xFF29 => [0xFF49], # Case map
          0xFF2A => [0xFF4A], # Case map
          0xFF2B => [0xFF4B], # Case map
          0xFF2C => [0xFF4C], # Case map
          0xFF2D => [0xFF4D], # Case map
          0xFF2E => [0xFF4E], # Case map
          0xFF2F => [0xFF4F], # Case map
          0xFF30 => [0xFF50], # Case map
          0xFF31 => [0xFF51], # Case map
          0xFF32 => [0xFF52], # Case map
          0xFF33 => [0xFF53], # Case map
          0xFF34 => [0xFF54], # Case map
          0xFF35 => [0xFF55], # Case map
          0xFF36 => [0xFF56], # Case map
          0xFF37 => [0xFF57], # Case map
          0xFF38 => [0xFF58], # Case map
          0xFF39 => [0xFF59], # Case map
          0xFF3A => [0xFF5A], # Case map
          0x10400 => [0x10428], # Case map
          0x10401 => [0x10429], # Case map
          0x10402 => [0x1042A], # Case map
          0x10403 => [0x1042B], # Case map
          0x10404 => [0x1042C], # Case map
          0x10405 => [0x1042D], # Case map
          0x10406 => [0x1042E], # Case map
          0x10407 => [0x1042F], # Case map
          0x10408 => [0x10430], # Case map
          0x10409 => [0x10431], # Case map
          0x1040A => [0x10432], # Case map
          0x1040B => [0x10433], # Case map
          0x1040C => [0x10434], # Case map
          0x1040D => [0x10435], # Case map
          0x1040E => [0x10436], # Case map
          0x1040F => [0x10437], # Case map
          0x10410 => [0x10438], # Case map
          0x10411 => [0x10439], # Case map
          0x10412 => [0x1043A], # Case map
          0x10413 => [0x1043B], # Case map
          0x10414 => [0x1043C], # Case map
          0x10415 => [0x1043D], # Case map
          0x10416 => [0x1043E], # Case map
          0x10417 => [0x1043F], # Case map
          0x10418 => [0x10440], # Case map
          0x10419 => [0x10441], # Case map
          0x1041A => [0x10442], # Case map
          0x1041B => [0x10443], # Case map
          0x1041C => [0x10444], # Case map
          0x1041D => [0x10445], # Case map
          0x1041E => [0x10446], # Case map
          0x1041F => [0x10447], # Case map
          0x10420 => [0x10448], # Case map
          0x10421 => [0x10449], # Case map
          0x10422 => [0x1044A], # Case map
          0x10423 => [0x1044B], # Case map
          0x10424 => [0x1044C], # Case map
          0x10425 => [0x1044D], # Case map
          0x1D400 => [0x0061], # Additional folding
          0x1D401 => [0x0062], # Additional folding
          0x1D402 => [0x0063], # Additional folding
          0x1D403 => [0x0064], # Additional folding
          0x1D404 => [0x0065], # Additional folding
          0x1D405 => [0x0066], # Additional folding
          0x1D406 => [0x0067], # Additional folding
          0x1D407 => [0x0068], # Additional folding
          0x1D408 => [0x0069], # Additional folding
          0x1D409 => [0x006A], # Additional folding
          0x1D40A => [0x006B], # Additional folding
          0x1D40B => [0x006C], # Additional folding
          0x1D40C => [0x006D], # Additional folding
          0x1D40D => [0x006E], # Additional folding
          0x1D40E => [0x006F], # Additional folding
          0x1D40F => [0x0070], # Additional folding
          0x1D410 => [0x0071], # Additional folding
          0x1D411 => [0x0072], # Additional folding
          0x1D412 => [0x0073], # Additional folding
          0x1D413 => [0x0074], # Additional folding
          0x1D414 => [0x0075], # Additional folding
          0x1D415 => [0x0076], # Additional folding
          0x1D416 => [0x0077], # Additional folding
          0x1D417 => [0x0078], # Additional folding
          0x1D418 => [0x0079], # Additional folding
          0x1D419 => [0x007A], # Additional folding
          0x1D434 => [0x0061], # Additional folding
          0x1D435 => [0x0062], # Additional folding
          0x1D436 => [0x0063], # Additional folding
          0x1D437 => [0x0064], # Additional folding
          0x1D438 => [0x0065], # Additional folding
          0x1D439 => [0x0066], # Additional folding
          0x1D43A => [0x0067], # Additional folding
          0x1D43B => [0x0068], # Additional folding
          0x1D43C => [0x0069], # Additional folding
          0x1D43D => [0x006A], # Additional folding
          0x1D43E => [0x006B], # Additional folding
          0x1D43F => [0x006C], # Additional folding
          0x1D440 => [0x006D], # Additional folding
          0x1D441 => [0x006E], # Additional folding
          0x1D442 => [0x006F], # Additional folding
          0x1D443 => [0x0070], # Additional folding
          0x1D444 => [0x0071], # Additional folding
          0x1D445 => [0x0072], # Additional folding
          0x1D446 => [0x0073], # Additional folding
          0x1D447 => [0x0074], # Additional folding
          0x1D448 => [0x0075], # Additional folding
          0x1D449 => [0x0076], # Additional folding
          0x1D44A => [0x0077], # Additional folding
          0x1D44B => [0x0078], # Additional folding
          0x1D44C => [0x0079], # Additional folding
          0x1D44D => [0x007A], # Additional folding
          0x1D468 => [0x0061], # Additional folding
          0x1D469 => [0x0062], # Additional folding
          0x1D46A => [0x0063], # Additional folding
          0x1D46B => [0x0064], # Additional folding
          0x1D46C => [0x0065], # Additional folding
          0x1D46D => [0x0066], # Additional folding
          0x1D46E => [0x0067], # Additional folding
          0x1D46F => [0x0068], # Additional folding
          0x1D470 => [0x0069], # Additional folding
          0x1D471 => [0x006A], # Additional folding
          0x1D472 => [0x006B], # Additional folding
          0x1D473 => [0x006C], # Additional folding
          0x1D474 => [0x006D], # Additional folding
          0x1D475 => [0x006E], # Additional folding
          0x1D476 => [0x006F], # Additional folding
          0x1D477 => [0x0070], # Additional folding
          0x1D478 => [0x0071], # Additional folding
          0x1D479 => [0x0072], # Additional folding
          0x1D47A => [0x0073], # Additional folding
          0x1D47B => [0x0074], # Additional folding
          0x1D47C => [0x0075], # Additional folding
          0x1D47D => [0x0076], # Additional folding
          0x1D47E => [0x0077], # Additional folding
          0x1D47F => [0x0078], # Additional folding
          0x1D480 => [0x0079], # Additional folding
          0x1D481 => [0x007A], # Additional folding
          0x1D49C => [0x0061], # Additional folding
          0x1D49E => [0x0063], # Additional folding
          0x1D49F => [0x0064], # Additional folding
          0x1D4A2 => [0x0067], # Additional folding
          0x1D4A5 => [0x006A], # Additional folding
          0x1D4A6 => [0x006B], # Additional folding
          0x1D4A9 => [0x006E], # Additional folding
          0x1D4AA => [0x006F], # Additional folding
          0x1D4AB => [0x0070], # Additional folding
          0x1D4AC => [0x0071], # Additional folding
          0x1D4AE => [0x0073], # Additional folding
          0x1D4AF => [0x0074], # Additional folding
          0x1D4B0 => [0x0075], # Additional folding
          0x1D4B1 => [0x0076], # Additional folding
          0x1D4B2 => [0x0077], # Additional folding
          0x1D4B3 => [0x0078], # Additional folding
          0x1D4B4 => [0x0079], # Additional folding
          0x1D4B5 => [0x007A], # Additional folding
          0x1D4D0 => [0x0061], # Additional folding
          0x1D4D1 => [0x0062], # Additional folding
          0x1D4D2 => [0x0063], # Additional folding
          0x1D4D3 => [0x0064], # Additional folding
          0x1D4D4 => [0x0065], # Additional folding
          0x1D4D5 => [0x0066], # Additional folding
          0x1D4D6 => [0x0067], # Additional folding
          0x1D4D7 => [0x0068], # Additional folding
          0x1D4D8 => [0x0069], # Additional folding
          0x1D4D9 => [0x006A], # Additional folding
          0x1D4DA => [0x006B], # Additional folding
          0x1D4DB => [0x006C], # Additional folding
          0x1D4DC => [0x006D], # Additional folding
          0x1D4DD => [0x006E], # Additional folding
          0x1D4DE => [0x006F], # Additional folding
          0x1D4DF => [0x0070], # Additional folding
          0x1D4E0 => [0x0071], # Additional folding
          0x1D4E1 => [0x0072], # Additional folding
          0x1D4E2 => [0x0073], # Additional folding
          0x1D4E3 => [0x0074], # Additional folding
          0x1D4E4 => [0x0075], # Additional folding
          0x1D4E5 => [0x0076], # Additional folding
          0x1D4E6 => [0x0077], # Additional folding
          0x1D4E7 => [0x0078], # Additional folding
          0x1D4E8 => [0x0079], # Additional folding
          0x1D4E9 => [0x007A], # Additional folding
          0x1D504 => [0x0061], # Additional folding
          0x1D505 => [0x0062], # Additional folding
          0x1D507 => [0x0064], # Additional folding
          0x1D508 => [0x0065], # Additional folding
          0x1D509 => [0x0066], # Additional folding
          0x1D50A => [0x0067], # Additional folding
          0x1D50D => [0x006A], # Additional folding
          0x1D50E => [0x006B], # Additional folding
          0x1D50F => [0x006C], # Additional folding
          0x1D510 => [0x006D], # Additional folding
          0x1D511 => [0x006E], # Additional folding
          0x1D512 => [0x006F], # Additional folding
          0x1D513 => [0x0070], # Additional folding
          0x1D514 => [0x0071], # Additional folding
          0x1D516 => [0x0073], # Additional folding
          0x1D517 => [0x0074], # Additional folding
          0x1D518 => [0x0075], # Additional folding
          0x1D519 => [0x0076], # Additional folding
          0x1D51A => [0x0077], # Additional folding
          0x1D51B => [0x0078], # Additional folding
          0x1D51C => [0x0079], # Additional folding
          0x1D538 => [0x0061], # Additional folding
          0x1D539 => [0x0062], # Additional folding
          0x1D53B => [0x0064], # Additional folding
          0x1D53C => [0x0065], # Additional folding
          0x1D53D => [0x0066], # Additional folding
          0x1D53E => [0x0067], # Additional folding
          0x1D540 => [0x0069], # Additional folding
          0x1D541 => [0x006A], # Additional folding
          0x1D542 => [0x006B], # Additional folding
          0x1D543 => [0x006C], # Additional folding
          0x1D544 => [0x006D], # Additional folding
          0x1D546 => [0x006F], # Additional folding
          0x1D54A => [0x0073], # Additional folding
          0x1D54B => [0x0074], # Additional folding
          0x1D54C => [0x0075], # Additional folding
          0x1D54D => [0x0076], # Additional folding
          0x1D54E => [0x0077], # Additional folding
          0x1D54F => [0x0078], # Additional folding
          0x1D550 => [0x0079], # Additional folding
          0x1D56C => [0x0061], # Additional folding
          0x1D56D => [0x0062], # Additional folding
          0x1D56E => [0x0063], # Additional folding
          0x1D56F => [0x0064], # Additional folding
          0x1D570 => [0x0065], # Additional folding
          0x1D571 => [0x0066], # Additional folding
          0x1D572 => [0x0067], # Additional folding
          0x1D573 => [0x0068], # Additional folding
          0x1D574 => [0x0069], # Additional folding
          0x1D575 => [0x006A], # Additional folding
          0x1D576 => [0x006B], # Additional folding
          0x1D577 => [0x006C], # Additional folding
          0x1D578 => [0x006D], # Additional folding
          0x1D579 => [0x006E], # Additional folding
          0x1D57A => [0x006F], # Additional folding
          0x1D57B => [0x0070], # Additional folding
          0x1D57C => [0x0071], # Additional folding
          0x1D57D => [0x0072], # Additional folding
          0x1D57E => [0x0073], # Additional folding
          0x1D57F => [0x0074], # Additional folding
          0x1D580 => [0x0075], # Additional folding
          0x1D581 => [0x0076], # Additional folding
          0x1D582 => [0x0077], # Additional folding
          0x1D583 => [0x0078], # Additional folding
          0x1D584 => [0x0079], # Additional folding
          0x1D585 => [0x007A], # Additional folding
          0x1D5A0 => [0x0061], # Additional folding
          0x1D5A1 => [0x0062], # Additional folding
          0x1D5A2 => [0x0063], # Additional folding
          0x1D5A3 => [0x0064], # Additional folding
          0x1D5A4 => [0x0065], # Additional folding
          0x1D5A5 => [0x0066], # Additional folding
          0x1D5A6 => [0x0067], # Additional folding
          0x1D5A7 => [0x0068], # Additional folding
          0x1D5A8 => [0x0069], # Additional folding
          0x1D5A9 => [0x006A], # Additional folding
          0x1D5AA => [0x006B], # Additional folding
          0x1D5AB => [0x006C], # Additional folding
          0x1D5AC => [0x006D], # Additional folding
          0x1D5AD => [0x006E], # Additional folding
          0x1D5AE => [0x006F], # Additional folding
          0x1D5AF => [0x0070], # Additional folding
          0x1D5B0 => [0x0071], # Additional folding
          0x1D5B1 => [0x0072], # Additional folding
          0x1D5B2 => [0x0073], # Additional folding
          0x1D5B3 => [0x0074], # Additional folding
          0x1D5B4 => [0x0075], # Additional folding
          0x1D5B5 => [0x0076], # Additional folding
          0x1D5B6 => [0x0077], # Additional folding
          0x1D5B7 => [0x0078], # Additional folding
          0x1D5B8 => [0x0079], # Additional folding
          0x1D5B9 => [0x007A], # Additional folding
          0x1D5D4 => [0x0061], # Additional folding
          0x1D5D5 => [0x0062], # Additional folding
          0x1D5D6 => [0x0063], # Additional folding
          0x1D5D7 => [0x0064], # Additional folding
          0x1D5D8 => [0x0065], # Additional folding
          0x1D5D9 => [0x0066], # Additional folding
          0x1D5DA => [0x0067], # Additional folding
          0x1D5DB => [0x0068], # Additional folding
          0x1D5DC => [0x0069], # Additional folding
          0x1D5DD => [0x006A], # Additional folding
          0x1D5DE => [0x006B], # Additional folding
          0x1D5DF => [0x006C], # Additional folding
          0x1D5E0 => [0x006D], # Additional folding
          0x1D5E1 => [0x006E], # Additional folding
          0x1D5E2 => [0x006F], # Additional folding
          0x1D5E3 => [0x0070], # Additional folding
          0x1D5E4 => [0x0071], # Additional folding
          0x1D5E5 => [0x0072], # Additional folding
          0x1D5E6 => [0x0073], # Additional folding
          0x1D5E7 => [0x0074], # Additional folding
          0x1D5E8 => [0x0075], # Additional folding
          0x1D5E9 => [0x0076], # Additional folding
          0x1D5EA => [0x0077], # Additional folding
          0x1D5EB => [0x0078], # Additional folding
          0x1D5EC => [0x0079], # Additional folding
          0x1D5ED => [0x007A], # Additional folding
          0x1D608 => [0x0061], # Additional folding
          0x1D609 => [0x0062], # Additional folding
          0x1D60A => [0x0063], # Additional folding
          0x1D60B => [0x0064], # Additional folding
          0x1D60C => [0x0065], # Additional folding
          0x1D60D => [0x0066], # Additional folding
          0x1D60E => [0x0067], # Additional folding
          0x1D60F => [0x0068], # Additional folding
          0x1D610 => [0x0069], # Additional folding
          0x1D611 => [0x006A], # Additional folding
          0x1D612 => [0x006B], # Additional folding
          0x1D613 => [0x006C], # Additional folding
          0x1D614 => [0x006D], # Additional folding
          0x1D615 => [0x006E], # Additional folding
          0x1D616 => [0x006F], # Additional folding
          0x1D617 => [0x0070], # Additional folding
          0x1D618 => [0x0071], # Additional folding
          0x1D619 => [0x0072], # Additional folding
          0x1D61A => [0x0073], # Additional folding
          0x1D61B => [0x0074], # Additional folding
          0x1D61C => [0x0075], # Additional folding
          0x1D61D => [0x0076], # Additional folding
          0x1D61E => [0x0077], # Additional folding
          0x1D61F => [0x0078], # Additional folding
          0x1D620 => [0x0079], # Additional folding
          0x1D621 => [0x007A], # Additional folding
          0x1D63C => [0x0061], # Additional folding
          0x1D63D => [0x0062], # Additional folding
          0x1D63E => [0x0063], # Additional folding
          0x1D63F => [0x0064], # Additional folding
          0x1D640 => [0x0065], # Additional folding
          0x1D641 => [0x0066], # Additional folding
          0x1D642 => [0x0067], # Additional folding
          0x1D643 => [0x0068], # Additional folding
          0x1D644 => [0x0069], # Additional folding
          0x1D645 => [0x006A], # Additional folding
          0x1D646 => [0x006B], # Additional folding
          0x1D647 => [0x006C], # Additional folding
          0x1D648 => [0x006D], # Additional folding
          0x1D649 => [0x006E], # Additional folding
          0x1D64A => [0x006F], # Additional folding
          0x1D64B => [0x0070], # Additional folding
          0x1D64C => [0x0071], # Additional folding
          0x1D64D => [0x0072], # Additional folding
          0x1D64E => [0x0073], # Additional folding
          0x1D64F => [0x0074], # Additional folding
          0x1D650 => [0x0075], # Additional folding
          0x1D651 => [0x0076], # Additional folding
          0x1D652 => [0x0077], # Additional folding
          0x1D653 => [0x0078], # Additional folding
          0x1D654 => [0x0079], # Additional folding
          0x1D655 => [0x007A], # Additional folding
          0x1D670 => [0x0061], # Additional folding
          0x1D671 => [0x0062], # Additional folding
          0x1D672 => [0x0063], # Additional folding
          0x1D673 => [0x0064], # Additional folding
          0x1D674 => [0x0065], # Additional folding
          0x1D675 => [0x0066], # Additional folding
          0x1D676 => [0x0067], # Additional folding
          0x1D677 => [0x0068], # Additional folding
          0x1D678 => [0x0069], # Additional folding
          0x1D679 => [0x006A], # Additional folding
          0x1D67A => [0x006B], # Additional folding
          0x1D67B => [0x006C], # Additional folding
          0x1D67C => [0x006D], # Additional folding
          0x1D67D => [0x006E], # Additional folding
          0x1D67E => [0x006F], # Additional folding
          0x1D67F => [0x0070], # Additional folding
          0x1D680 => [0x0071], # Additional folding
          0x1D681 => [0x0072], # Additional folding
          0x1D682 => [0x0073], # Additional folding
          0x1D683 => [0x0074], # Additional folding
          0x1D684 => [0x0075], # Additional folding
          0x1D685 => [0x0076], # Additional folding
          0x1D686 => [0x0077], # Additional folding
          0x1D687 => [0x0078], # Additional folding
          0x1D688 => [0x0079], # Additional folding
          0x1D689 => [0x007A], # Additional folding
          0x1D6A8 => [0x03B1], # Additional folding
          0x1D6A9 => [0x03B2], # Additional folding
          0x1D6AA => [0x03B3], # Additional folding
          0x1D6AB => [0x03B4], # Additional folding
          0x1D6AC => [0x03B5], # Additional folding
          0x1D6AD => [0x03B6], # Additional folding
          0x1D6AE => [0x03B7], # Additional folding
          0x1D6AF => [0x03B8], # Additional folding
          0x1D6B0 => [0x03B9], # Additional folding
          0x1D6B1 => [0x03BA], # Additional folding
          0x1D6B2 => [0x03BB], # Additional folding
          0x1D6B3 => [0x03BC], # Additional folding
          0x1D6B4 => [0x03BD], # Additional folding
          0x1D6B5 => [0x03BE], # Additional folding
          0x1D6B6 => [0x03BF], # Additional folding
          0x1D6B7 => [0x03C0], # Additional folding
          0x1D6B8 => [0x03C1], # Additional folding
          0x1D6B9 => [0x03B8], # Additional folding
          0x1D6BA => [0x03C3], # Additional folding
          0x1D6BB => [0x03C4], # Additional folding
          0x1D6BC => [0x03C5], # Additional folding
          0x1D6BD => [0x03C6], # Additional folding
          0x1D6BE => [0x03C7], # Additional folding
          0x1D6BF => [0x03C8], # Additional folding
          0x1D6C0 => [0x03C9], # Additional folding
          0x1D6D3 => [0x03C3], # Additional folding
          0x1D6E2 => [0x03B1], # Additional folding
          0x1D6E3 => [0x03B2], # Additional folding
          0x1D6E4 => [0x03B3], # Additional folding
          0x1D6E5 => [0x03B4], # Additional folding
          0x1D6E6 => [0x03B5], # Additional folding
          0x1D6E7 => [0x03B6], # Additional folding
          0x1D6E8 => [0x03B7], # Additional folding
          0x1D6E9 => [0x03B8], # Additional folding
          0x1D6EA => [0x03B9], # Additional folding
          0x1D6EB => [0x03BA], # Additional folding
          0x1D6EC => [0x03BB], # Additional folding
          0x1D6ED => [0x03BC], # Additional folding
          0x1D6EE => [0x03BD], # Additional folding
          0x1D6EF => [0x03BE], # Additional folding
          0x1D6F0 => [0x03BF], # Additional folding
          0x1D6F1 => [0x03C0], # Additional folding
          0x1D6F2 => [0x03C1], # Additional folding
          0x1D6F3 => [0x03B8], # Additional folding
          0x1D6F4 => [0x03C3], # Additional folding
          0x1D6F5 => [0x03C4], # Additional folding
          0x1D6F6 => [0x03C5], # Additional folding
          0x1D6F7 => [0x03C6], # Additional folding
          0x1D6F8 => [0x03C7], # Additional folding
          0x1D6F9 => [0x03C8], # Additional folding
          0x1D6FA => [0x03C9], # Additional folding
          0x1D70D => [0x03C3], # Additional folding
          0x1D71C => [0x03B1], # Additional folding
          0x1D71D => [0x03B2], # Additional folding
          0x1D71E => [0x03B3], # Additional folding
          0x1D71F => [0x03B4], # Additional folding
          0x1D720 => [0x03B5], # Additional folding
          0x1D721 => [0x03B6], # Additional folding
          0x1D722 => [0x03B7], # Additional folding
          0x1D723 => [0x03B8], # Additional folding
          0x1D724 => [0x03B9], # Additional folding
          0x1D725 => [0x03BA], # Additional folding
          0x1D726 => [0x03BB], # Additional folding
          0x1D727 => [0x03BC], # Additional folding
          0x1D728 => [0x03BD], # Additional folding
          0x1D729 => [0x03BE], # Additional folding
          0x1D72A => [0x03BF], # Additional folding
          0x1D72B => [0x03C0], # Additional folding
          0x1D72C => [0x03C1], # Additional folding
          0x1D72D => [0x03B8], # Additional folding
          0x1D72E => [0x03C3], # Additional folding
          0x1D72F => [0x03C4], # Additional folding
          0x1D730 => [0x03C5], # Additional folding
          0x1D731 => [0x03C6], # Additional folding
          0x1D732 => [0x03C7], # Additional folding
          0x1D733 => [0x03C8], # Additional folding
          0x1D734 => [0x03C9], # Additional folding
          0x1D747 => [0x03C3], # Additional folding
          0x1D756 => [0x03B1], # Additional folding
          0x1D757 => [0x03B2], # Additional folding
          0x1D758 => [0x03B3], # Additional folding
          0x1D759 => [0x03B4], # Additional folding
          0x1D75A => [0x03B5], # Additional folding
          0x1D75B => [0x03B6], # Additional folding
          0x1D75C => [0x03B7], # Additional folding
          0x1D75D => [0x03B8], # Additional folding
          0x1D75E => [0x03B9], # Additional folding
          0x1D75F => [0x03BA], # Additional folding
          0x1D760 => [0x03BB], # Additional folding
          0x1D761 => [0x03BC], # Additional folding
          0x1D762 => [0x03BD], # Additional folding
          0x1D763 => [0x03BE], # Additional folding
          0x1D764 => [0x03BF], # Additional folding
          0x1D765 => [0x03C0], # Additional folding
          0x1D766 => [0x03C1], # Additional folding
          0x1D767 => [0x03B8], # Additional folding
          0x1D768 => [0x03C3], # Additional folding
          0x1D769 => [0x03C4], # Additional folding
          0x1D76A => [0x03C5], # Additional folding
          0x1D76B => [0x03C6], # Additional folding
          0x1D76C => [0x03C7], # Additional folding
          0x1D76D => [0x03C8], # Additional folding
          0x1D76E => [0x03C9], # Additional folding
          0x1D781 => [0x03C3], # Additional folding
          0x1D790 => [0x03B1], # Additional folding
          0x1D791 => [0x03B2], # Additional folding
          0x1D792 => [0x03B3], # Additional folding
          0x1D793 => [0x03B4], # Additional folding
          0x1D794 => [0x03B5], # Additional folding
          0x1D795 => [0x03B6], # Additional folding
          0x1D796 => [0x03B7], # Additional folding
          0x1D797 => [0x03B8], # Additional folding
          0x1D798 => [0x03B9], # Additional folding
          0x1D799 => [0x03BA], # Additional folding
          0x1D79A => [0x03BB], # Additional folding
          0x1D79B => [0x03BC], # Additional folding
          0x1D79C => [0x03BD], # Additional folding
          0x1D79D => [0x03BE], # Additional folding
          0x1D79E => [0x03BF], # Additional folding
          0x1D79F => [0x03C0], # Additional folding
          0x1D7A0 => [0x03C1], # Additional folding
          0x1D7A1 => [0x03B8], # Additional folding
          0x1D7A2 => [0x03C3], # Additional folding
          0x1D7A3 => [0x03C4], # Additional folding
          0x1D7A4 => [0x03C5], # Additional folding
          0x1D7A5 => [0x03C6], # Additional folding
          0x1D7A6 => [0x03C7], # Additional folding
          0x1D7A7 => [0x03C8], # Additional folding
          0x1D7A8 => [0x03C9], # Additional folding
          0x1D7BB => [0x03C3], # Additional folding
        }.freeze

        # Table B3 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        B3 = {
          0x0041 => [0x0061], # Case map
          0x0042 => [0x0062], # Case map
          0x0043 => [0x0063], # Case map
          0x0044 => [0x0064], # Case map
          0x0045 => [0x0065], # Case map
          0x0046 => [0x0066], # Case map
          0x0047 => [0x0067], # Case map
          0x0048 => [0x0068], # Case map
          0x0049 => [0x0069], # Case map
          0x004A => [0x006A], # Case map
          0x004B => [0x006B], # Case map
          0x004C => [0x006C], # Case map
          0x004D => [0x006D], # Case map
          0x004E => [0x006E], # Case map
          0x004F => [0x006F], # Case map
          0x0050 => [0x0070], # Case map
          0x0051 => [0x0071], # Case map
          0x0052 => [0x0072], # Case map
          0x0053 => [0x0073], # Case map
          0x0054 => [0x0074], # Case map
          0x0055 => [0x0075], # Case map
          0x0056 => [0x0076], # Case map
          0x0057 => [0x0077], # Case map
          0x0058 => [0x0078], # Case map
          0x0059 => [0x0079], # Case map
          0x005A => [0x007A], # Case map
          0x00B5 => [0x03BC], # Case map
          0x00C0 => [0x00E0], # Case map
          0x00C1 => [0x00E1], # Case map
          0x00C2 => [0x00E2], # Case map
          0x00C3 => [0x00E3], # Case map
          0x00C4 => [0x00E4], # Case map
          0x00C5 => [0x00E5], # Case map
          0x00C6 => [0x00E6], # Case map
          0x00C7 => [0x00E7], # Case map
          0x00C8 => [0x00E8], # Case map
          0x00C9 => [0x00E9], # Case map
          0x00CA => [0x00EA], # Case map
          0x00CB => [0x00EB], # Case map
          0x00CC => [0x00EC], # Case map
          0x00CD => [0x00ED], # Case map
          0x00CE => [0x00EE], # Case map
          0x00CF => [0x00EF], # Case map
          0x00D0 => [0x00F0], # Case map
          0x00D1 => [0x00F1], # Case map
          0x00D2 => [0x00F2], # Case map
          0x00D3 => [0x00F3], # Case map
          0x00D4 => [0x00F4], # Case map
          0x00D5 => [0x00F5], # Case map
          0x00D6 => [0x00F6], # Case map
          0x00D8 => [0x00F8], # Case map
          0x00D9 => [0x00F9], # Case map
          0x00DA => [0x00FA], # Case map
          0x00DB => [0x00FB], # Case map
          0x00DC => [0x00FC], # Case map
          0x00DD => [0x00FD], # Case map
          0x00DE => [0x00FE], # Case map
          0x00DF => [0x0073, 0x0073], # Case map
          0x0100 => [0x0101], # Case map
          0x0102 => [0x0103], # Case map
          0x0104 => [0x0105], # Case map
          0x0106 => [0x0107], # Case map
          0x0108 => [0x0109], # Case map
          0x010A => [0x010B], # Case map
          0x010C => [0x010D], # Case map
          0x010E => [0x010F], # Case map
          0x0110 => [0x0111], # Case map
          0x0112 => [0x0113], # Case map
          0x0114 => [0x0115], # Case map
          0x0116 => [0x0117], # Case map
          0x0118 => [0x0119], # Case map
          0x011A => [0x011B], # Case map
          0x011C => [0x011D], # Case map
          0x011E => [0x011F], # Case map
          0x0120 => [0x0121], # Case map
          0x0122 => [0x0123], # Case map
          0x0124 => [0x0125], # Case map
          0x0126 => [0x0127], # Case map
          0x0128 => [0x0129], # Case map
          0x012A => [0x012B], # Case map
          0x012C => [0x012D], # Case map
          0x012E => [0x012F], # Case map
          0x0130 => [0x0069, 0x0307], # Case map
          0x0132 => [0x0133], # Case map
          0x0134 => [0x0135], # Case map
          0x0136 => [0x0137], # Case map
          0x0139 => [0x013A], # Case map
          0x013B => [0x013C], # Case map
          0x013D => [0x013E], # Case map
          0x013F => [0x0140], # Case map
          0x0141 => [0x0142], # Case map
          0x0143 => [0x0144], # Case map
          0x0145 => [0x0146], # Case map
          0x0147 => [0x0148], # Case map
          0x0149 => [0x02BC, 0x006E], # Case map
          0x014A => [0x014B], # Case map
          0x014C => [0x014D], # Case map
          0x014E => [0x014F], # Case map
          0x0150 => [0x0151], # Case map
          0x0152 => [0x0153], # Case map
          0x0154 => [0x0155], # Case map
          0x0156 => [0x0157], # Case map
          0x0158 => [0x0159], # Case map
          0x015A => [0x015B], # Case map
          0x015C => [0x015D], # Case map
          0x015E => [0x015F], # Case map
          0x0160 => [0x0161], # Case map
          0x0162 => [0x0163], # Case map
          0x0164 => [0x0165], # Case map
          0x0166 => [0x0167], # Case map
          0x0168 => [0x0169], # Case map
          0x016A => [0x016B], # Case map
          0x016C => [0x016D], # Case map
          0x016E => [0x016F], # Case map
          0x0170 => [0x0171], # Case map
          0x0172 => [0x0173], # Case map
          0x0174 => [0x0175], # Case map
          0x0176 => [0x0177], # Case map
          0x0178 => [0x00FF], # Case map
          0x0179 => [0x017A], # Case map
          0x017B => [0x017C], # Case map
          0x017D => [0x017E], # Case map
          0x017F => [0x0073], # Case map
          0x0181 => [0x0253], # Case map
          0x0182 => [0x0183], # Case map
          0x0184 => [0x0185], # Case map
          0x0186 => [0x0254], # Case map
          0x0187 => [0x0188], # Case map
          0x0189 => [0x0256], # Case map
          0x018A => [0x0257], # Case map
          0x018B => [0x018C], # Case map
          0x018E => [0x01DD], # Case map
          0x018F => [0x0259], # Case map
          0x0190 => [0x025B], # Case map
          0x0191 => [0x0192], # Case map
          0x0193 => [0x0260], # Case map
          0x0194 => [0x0263], # Case map
          0x0196 => [0x0269], # Case map
          0x0197 => [0x0268], # Case map
          0x0198 => [0x0199], # Case map
          0x019C => [0x026F], # Case map
          0x019D => [0x0272], # Case map
          0x019F => [0x0275], # Case map
          0x01A0 => [0x01A1], # Case map
          0x01A2 => [0x01A3], # Case map
          0x01A4 => [0x01A5], # Case map
          0x01A6 => [0x0280], # Case map
          0x01A7 => [0x01A8], # Case map
          0x01A9 => [0x0283], # Case map
          0x01AC => [0x01AD], # Case map
          0x01AE => [0x0288], # Case map
          0x01AF => [0x01B0], # Case map
          0x01B1 => [0x028A], # Case map
          0x01B2 => [0x028B], # Case map
          0x01B3 => [0x01B4], # Case map
          0x01B5 => [0x01B6], # Case map
          0x01B7 => [0x0292], # Case map
          0x01B8 => [0x01B9], # Case map
          0x01BC => [0x01BD], # Case map
          0x01C4 => [0x01C6], # Case map
          0x01C5 => [0x01C6], # Case map
          0x01C7 => [0x01C9], # Case map
          0x01C8 => [0x01C9], # Case map
          0x01CA => [0x01CC], # Case map
          0x01CB => [0x01CC], # Case map
          0x01CD => [0x01CE], # Case map
          0x01CF => [0x01D0], # Case map
          0x01D1 => [0x01D2], # Case map
          0x01D3 => [0x01D4], # Case map
          0x01D5 => [0x01D6], # Case map
          0x01D7 => [0x01D8], # Case map
          0x01D9 => [0x01DA], # Case map
          0x01DB => [0x01DC], # Case map
          0x01DE => [0x01DF], # Case map
          0x01E0 => [0x01E1], # Case map
          0x01E2 => [0x01E3], # Case map
          0x01E4 => [0x01E5], # Case map
          0x01E6 => [0x01E7], # Case map
          0x01E8 => [0x01E9], # Case map
          0x01EA => [0x01EB], # Case map
          0x01EC => [0x01ED], # Case map
          0x01EE => [0x01EF], # Case map
          0x01F0 => [0x006A, 0x030C], # Case map
          0x01F1 => [0x01F3], # Case map
          0x01F2 => [0x01F3], # Case map
          0x01F4 => [0x01F5], # Case map
          0x01F6 => [0x0195], # Case map
          0x01F7 => [0x01BF], # Case map
          0x01F8 => [0x01F9], # Case map
          0x01FA => [0x01FB], # Case map
          0x01FC => [0x01FD], # Case map
          0x01FE => [0x01FF], # Case map
          0x0200 => [0x0201], # Case map
          0x0202 => [0x0203], # Case map
          0x0204 => [0x0205], # Case map
          0x0206 => [0x0207], # Case map
          0x0208 => [0x0209], # Case map
          0x020A => [0x020B], # Case map
          0x020C => [0x020D], # Case map
          0x020E => [0x020F], # Case map
          0x0210 => [0x0211], # Case map
          0x0212 => [0x0213], # Case map
          0x0214 => [0x0215], # Case map
          0x0216 => [0x0217], # Case map
          0x0218 => [0x0219], # Case map
          0x021A => [0x021B], # Case map
          0x021C => [0x021D], # Case map
          0x021E => [0x021F], # Case map
          0x0220 => [0x019E], # Case map
          0x0222 => [0x0223], # Case map
          0x0224 => [0x0225], # Case map
          0x0226 => [0x0227], # Case map
          0x0228 => [0x0229], # Case map
          0x022A => [0x022B], # Case map
          0x022C => [0x022D], # Case map
          0x022E => [0x022F], # Case map
          0x0230 => [0x0231], # Case map
          0x0232 => [0x0233], # Case map
          0x0345 => [0x03B9], # Case map
          0x0386 => [0x03AC], # Case map
          0x0388 => [0x03AD], # Case map
          0x0389 => [0x03AE], # Case map
          0x038A => [0x03AF], # Case map
          0x038C => [0x03CC], # Case map
          0x038E => [0x03CD], # Case map
          0x038F => [0x03CE], # Case map
          0x0390 => [0x03B9, 0x0308, 0x0301], # Case map
          0x0391 => [0x03B1], # Case map
          0x0392 => [0x03B2], # Case map
          0x0393 => [0x03B3], # Case map
          0x0394 => [0x03B4], # Case map
          0x0395 => [0x03B5], # Case map
          0x0396 => [0x03B6], # Case map
          0x0397 => [0x03B7], # Case map
          0x0398 => [0x03B8], # Case map
          0x0399 => [0x03B9], # Case map
          0x039A => [0x03BA], # Case map
          0x039B => [0x03BB], # Case map
          0x039C => [0x03BC], # Case map
          0x039D => [0x03BD], # Case map
          0x039E => [0x03BE], # Case map
          0x039F => [0x03BF], # Case map
          0x03A0 => [0x03C0], # Case map
          0x03A1 => [0x03C1], # Case map
          0x03A3 => [0x03C3], # Case map
          0x03A4 => [0x03C4], # Case map
          0x03A5 => [0x03C5], # Case map
          0x03A6 => [0x03C6], # Case map
          0x03A7 => [0x03C7], # Case map
          0x03A8 => [0x03C8], # Case map
          0x03A9 => [0x03C9], # Case map
          0x03AA => [0x03CA], # Case map
          0x03AB => [0x03CB], # Case map
          0x03B0 => [0x03C5, 0x0308, 0x0301], # Case map
          0x03C2 => [0x03C3], # Case map
          0x03D0 => [0x03B2], # Case map
          0x03D1 => [0x03B8], # Case map
          0x03D5 => [0x03C6], # Case map
          0x03D6 => [0x03C0], # Case map
          0x03D8 => [0x03D9], # Case map
          0x03DA => [0x03DB], # Case map
          0x03DC => [0x03DD], # Case map
          0x03DE => [0x03DF], # Case map
          0x03E0 => [0x03E1], # Case map
          0x03E2 => [0x03E3], # Case map
          0x03E4 => [0x03E5], # Case map
          0x03E6 => [0x03E7], # Case map
          0x03E8 => [0x03E9], # Case map
          0x03EA => [0x03EB], # Case map
          0x03EC => [0x03ED], # Case map
          0x03EE => [0x03EF], # Case map
          0x03F0 => [0x03BA], # Case map
          0x03F1 => [0x03C1], # Case map
          0x03F2 => [0x03C3], # Case map
          0x03F4 => [0x03B8], # Case map
          0x03F5 => [0x03B5], # Case map
          0x0400 => [0x0450], # Case map
          0x0401 => [0x0451], # Case map
          0x0402 => [0x0452], # Case map
          0x0403 => [0x0453], # Case map
          0x0404 => [0x0454], # Case map
          0x0405 => [0x0455], # Case map
          0x0406 => [0x0456], # Case map
          0x0407 => [0x0457], # Case map
          0x0408 => [0x0458], # Case map
          0x0409 => [0x0459], # Case map
          0x040A => [0x045A], # Case map
          0x040B => [0x045B], # Case map
          0x040C => [0x045C], # Case map
          0x040D => [0x045D], # Case map
          0x040E => [0x045E], # Case map
          0x040F => [0x045F], # Case map
          0x0410 => [0x0430], # Case map
          0x0411 => [0x0431], # Case map
          0x0412 => [0x0432], # Case map
          0x0413 => [0x0433], # Case map
          0x0414 => [0x0434], # Case map
          0x0415 => [0x0435], # Case map
          0x0416 => [0x0436], # Case map
          0x0417 => [0x0437], # Case map
          0x0418 => [0x0438], # Case map
          0x0419 => [0x0439], # Case map
          0x041A => [0x043A], # Case map
          0x041B => [0x043B], # Case map
          0x041C => [0x043C], # Case map
          0x041D => [0x043D], # Case map
          0x041E => [0x043E], # Case map
          0x041F => [0x043F], # Case map
          0x0420 => [0x0440], # Case map
          0x0421 => [0x0441], # Case map
          0x0422 => [0x0442], # Case map
          0x0423 => [0x0443], # Case map
          0x0424 => [0x0444], # Case map
          0x0425 => [0x0445], # Case map
          0x0426 => [0x0446], # Case map
          0x0427 => [0x0447], # Case map
          0x0428 => [0x0448], # Case map
          0x0429 => [0x0449], # Case map
          0x042A => [0x044A], # Case map
          0x042B => [0x044B], # Case map
          0x042C => [0x044C], # Case map
          0x042D => [0x044D], # Case map
          0x042E => [0x044E], # Case map
          0x042F => [0x044F], # Case map
          0x0460 => [0x0461], # Case map
          0x0462 => [0x0463], # Case map
          0x0464 => [0x0465], # Case map
          0x0466 => [0x0467], # Case map
          0x0468 => [0x0469], # Case map
          0x046A => [0x046B], # Case map
          0x046C => [0x046D], # Case map
          0x046E => [0x046F], # Case map
          0x0470 => [0x0471], # Case map
          0x0472 => [0x0473], # Case map
          0x0474 => [0x0475], # Case map
          0x0476 => [0x0477], # Case map
          0x0478 => [0x0479], # Case map
          0x047A => [0x047B], # Case map
          0x047C => [0x047D], # Case map
          0x047E => [0x047F], # Case map
          0x0480 => [0x0481], # Case map
          0x048A => [0x048B], # Case map
          0x048C => [0x048D], # Case map
          0x048E => [0x048F], # Case map
          0x0490 => [0x0491], # Case map
          0x0492 => [0x0493], # Case map
          0x0494 => [0x0495], # Case map
          0x0496 => [0x0497], # Case map
          0x0498 => [0x0499], # Case map
          0x049A => [0x049B], # Case map
          0x049C => [0x049D], # Case map
          0x049E => [0x049F], # Case map
          0x04A0 => [0x04A1], # Case map
          0x04A2 => [0x04A3], # Case map
          0x04A4 => [0x04A5], # Case map
          0x04A6 => [0x04A7], # Case map
          0x04A8 => [0x04A9], # Case map
          0x04AA => [0x04AB], # Case map
          0x04AC => [0x04AD], # Case map
          0x04AE => [0x04AF], # Case map
          0x04B0 => [0x04B1], # Case map
          0x04B2 => [0x04B3], # Case map
          0x04B4 => [0x04B5], # Case map
          0x04B6 => [0x04B7], # Case map
          0x04B8 => [0x04B9], # Case map
          0x04BA => [0x04BB], # Case map
          0x04BC => [0x04BD], # Case map
          0x04BE => [0x04BF], # Case map
          0x04C1 => [0x04C2], # Case map
          0x04C3 => [0x04C4], # Case map
          0x04C5 => [0x04C6], # Case map
          0x04C7 => [0x04C8], # Case map
          0x04C9 => [0x04CA], # Case map
          0x04CB => [0x04CC], # Case map
          0x04CD => [0x04CE], # Case map
          0x04D0 => [0x04D1], # Case map
          0x04D2 => [0x04D3], # Case map
          0x04D4 => [0x04D5], # Case map
          0x04D6 => [0x04D7], # Case map
          0x04D8 => [0x04D9], # Case map
          0x04DA => [0x04DB], # Case map
          0x04DC => [0x04DD], # Case map
          0x04DE => [0x04DF], # Case map
          0x04E0 => [0x04E1], # Case map
          0x04E2 => [0x04E3], # Case map
          0x04E4 => [0x04E5], # Case map
          0x04E6 => [0x04E7], # Case map
          0x04E8 => [0x04E9], # Case map
          0x04EA => [0x04EB], # Case map
          0x04EC => [0x04ED], # Case map
          0x04EE => [0x04EF], # Case map
          0x04F0 => [0x04F1], # Case map
          0x04F2 => [0x04F3], # Case map
          0x04F4 => [0x04F5], # Case map
          0x04F8 => [0x04F9], # Case map
          0x0500 => [0x0501], # Case map
          0x0502 => [0x0503], # Case map
          0x0504 => [0x0505], # Case map
          0x0506 => [0x0507], # Case map
          0x0508 => [0x0509], # Case map
          0x050A => [0x050B], # Case map
          0x050C => [0x050D], # Case map
          0x050E => [0x050F], # Case map
          0x0531 => [0x0561], # Case map
          0x0532 => [0x0562], # Case map
          0x0533 => [0x0563], # Case map
          0x0534 => [0x0564], # Case map
          0x0535 => [0x0565], # Case map
          0x0536 => [0x0566], # Case map
          0x0537 => [0x0567], # Case map
          0x0538 => [0x0568], # Case map
          0x0539 => [0x0569], # Case map
          0x053A => [0x056A], # Case map
          0x053B => [0x056B], # Case map
          0x053C => [0x056C], # Case map
          0x053D => [0x056D], # Case map
          0x053E => [0x056E], # Case map
          0x053F => [0x056F], # Case map
          0x0540 => [0x0570], # Case map
          0x0541 => [0x0571], # Case map
          0x0542 => [0x0572], # Case map
          0x0543 => [0x0573], # Case map
          0x0544 => [0x0574], # Case map
          0x0545 => [0x0575], # Case map
          0x0546 => [0x0576], # Case map
          0x0547 => [0x0577], # Case map
          0x0548 => [0x0578], # Case map
          0x0549 => [0x0579], # Case map
          0x054A => [0x057A], # Case map
          0x054B => [0x057B], # Case map
          0x054C => [0x057C], # Case map
          0x054D => [0x057D], # Case map
          0x054E => [0x057E], # Case map
          0x054F => [0x057F], # Case map
          0x0550 => [0x0580], # Case map
          0x0551 => [0x0581], # Case map
          0x0552 => [0x0582], # Case map
          0x0553 => [0x0583], # Case map
          0x0554 => [0x0584], # Case map
          0x0555 => [0x0585], # Case map
          0x0556 => [0x0586], # Case map
          0x0587 => [0x0565, 0x0582], # Case map
          0x1E00 => [0x1E01], # Case map
          0x1E02 => [0x1E03], # Case map
          0x1E04 => [0x1E05], # Case map
          0x1E06 => [0x1E07], # Case map
          0x1E08 => [0x1E09], # Case map
          0x1E0A => [0x1E0B], # Case map
          0x1E0C => [0x1E0D], # Case map
          0x1E0E => [0x1E0F], # Case map
          0x1E10 => [0x1E11], # Case map
          0x1E12 => [0x1E13], # Case map
          0x1E14 => [0x1E15], # Case map
          0x1E16 => [0x1E17], # Case map
          0x1E18 => [0x1E19], # Case map
          0x1E1A => [0x1E1B], # Case map
          0x1E1C => [0x1E1D], # Case map
          0x1E1E => [0x1E1F], # Case map
          0x1E20 => [0x1E21], # Case map
          0x1E22 => [0x1E23], # Case map
          0x1E24 => [0x1E25], # Case map
          0x1E26 => [0x1E27], # Case map
          0x1E28 => [0x1E29], # Case map
          0x1E2A => [0x1E2B], # Case map
          0x1E2C => [0x1E2D], # Case map
          0x1E2E => [0x1E2F], # Case map
          0x1E30 => [0x1E31], # Case map
          0x1E32 => [0x1E33], # Case map
          0x1E34 => [0x1E35], # Case map
          0x1E36 => [0x1E37], # Case map
          0x1E38 => [0x1E39], # Case map
          0x1E3A => [0x1E3B], # Case map
          0x1E3C => [0x1E3D], # Case map
          0x1E3E => [0x1E3F], # Case map
          0x1E40 => [0x1E41], # Case map
          0x1E42 => [0x1E43], # Case map
          0x1E44 => [0x1E45], # Case map
          0x1E46 => [0x1E47], # Case map
          0x1E48 => [0x1E49], # Case map
          0x1E4A => [0x1E4B], # Case map
          0x1E4C => [0x1E4D], # Case map
          0x1E4E => [0x1E4F], # Case map
          0x1E50 => [0x1E51], # Case map
          0x1E52 => [0x1E53], # Case map
          0x1E54 => [0x1E55], # Case map
          0x1E56 => [0x1E57], # Case map
          0x1E58 => [0x1E59], # Case map
          0x1E5A => [0x1E5B], # Case map
          0x1E5C => [0x1E5D], # Case map
          0x1E5E => [0x1E5F], # Case map
          0x1E60 => [0x1E61], # Case map
          0x1E62 => [0x1E63], # Case map
          0x1E64 => [0x1E65], # Case map
          0x1E66 => [0x1E67], # Case map
          0x1E68 => [0x1E69], # Case map
          0x1E6A => [0x1E6B], # Case map
          0x1E6C => [0x1E6D], # Case map
          0x1E6E => [0x1E6F], # Case map
          0x1E70 => [0x1E71], # Case map
          0x1E72 => [0x1E73], # Case map
          0x1E74 => [0x1E75], # Case map
          0x1E76 => [0x1E77], # Case map
          0x1E78 => [0x1E79], # Case map
          0x1E7A => [0x1E7B], # Case map
          0x1E7C => [0x1E7D], # Case map
          0x1E7E => [0x1E7F], # Case map
          0x1E80 => [0x1E81], # Case map
          0x1E82 => [0x1E83], # Case map
          0x1E84 => [0x1E85], # Case map
          0x1E86 => [0x1E87], # Case map
          0x1E88 => [0x1E89], # Case map
          0x1E8A => [0x1E8B], # Case map
          0x1E8C => [0x1E8D], # Case map
          0x1E8E => [0x1E8F], # Case map
          0x1E90 => [0x1E91], # Case map
          0x1E92 => [0x1E93], # Case map
          0x1E94 => [0x1E95], # Case map
          0x1E96 => [0x0068, 0x0331], # Case map
          0x1E97 => [0x0074, 0x0308], # Case map
          0x1E98 => [0x0077, 0x030A], # Case map
          0x1E99 => [0x0079, 0x030A], # Case map
          0x1E9A => [0x0061, 0x02BE], # Case map
          0x1E9B => [0x1E61], # Case map
          0x1EA0 => [0x1EA1], # Case map
          0x1EA2 => [0x1EA3], # Case map
          0x1EA4 => [0x1EA5], # Case map
          0x1EA6 => [0x1EA7], # Case map
          0x1EA8 => [0x1EA9], # Case map
          0x1EAA => [0x1EAB], # Case map
          0x1EAC => [0x1EAD], # Case map
          0x1EAE => [0x1EAF], # Case map
          0x1EB0 => [0x1EB1], # Case map
          0x1EB2 => [0x1EB3], # Case map
          0x1EB4 => [0x1EB5], # Case map
          0x1EB6 => [0x1EB7], # Case map
          0x1EB8 => [0x1EB9], # Case map
          0x1EBA => [0x1EBB], # Case map
          0x1EBC => [0x1EBD], # Case map
          0x1EBE => [0x1EBF], # Case map
          0x1EC0 => [0x1EC1], # Case map
          0x1EC2 => [0x1EC3], # Case map
          0x1EC4 => [0x1EC5], # Case map
          0x1EC6 => [0x1EC7], # Case map
          0x1EC8 => [0x1EC9], # Case map
          0x1ECA => [0x1ECB], # Case map
          0x1ECC => [0x1ECD], # Case map
          0x1ECE => [0x1ECF], # Case map
          0x1ED0 => [0x1ED1], # Case map
          0x1ED2 => [0x1ED3], # Case map
          0x1ED4 => [0x1ED5], # Case map
          0x1ED6 => [0x1ED7], # Case map
          0x1ED8 => [0x1ED9], # Case map
          0x1EDA => [0x1EDB], # Case map
          0x1EDC => [0x1EDD], # Case map
          0x1EDE => [0x1EDF], # Case map
          0x1EE0 => [0x1EE1], # Case map
          0x1EE2 => [0x1EE3], # Case map
          0x1EE4 => [0x1EE5], # Case map
          0x1EE6 => [0x1EE7], # Case map
          0x1EE8 => [0x1EE9], # Case map
          0x1EEA => [0x1EEB], # Case map
          0x1EEC => [0x1EED], # Case map
          0x1EEE => [0x1EEF], # Case map
          0x1EF0 => [0x1EF1], # Case map
          0x1EF2 => [0x1EF3], # Case map
          0x1EF4 => [0x1EF5], # Case map
          0x1EF6 => [0x1EF7], # Case map
          0x1EF8 => [0x1EF9], # Case map
          0x1F08 => [0x1F00], # Case map
          0x1F09 => [0x1F01], # Case map
          0x1F0A => [0x1F02], # Case map
          0x1F0B => [0x1F03], # Case map
          0x1F0C => [0x1F04], # Case map
          0x1F0D => [0x1F05], # Case map
          0x1F0E => [0x1F06], # Case map
          0x1F0F => [0x1F07], # Case map
          0x1F18 => [0x1F10], # Case map
          0x1F19 => [0x1F11], # Case map
          0x1F1A => [0x1F12], # Case map
          0x1F1B => [0x1F13], # Case map
          0x1F1C => [0x1F14], # Case map
          0x1F1D => [0x1F15], # Case map
          0x1F28 => [0x1F20], # Case map
          0x1F29 => [0x1F21], # Case map
          0x1F2A => [0x1F22], # Case map
          0x1F2B => [0x1F23], # Case map
          0x1F2C => [0x1F24], # Case map
          0x1F2D => [0x1F25], # Case map
          0x1F2E => [0x1F26], # Case map
          0x1F2F => [0x1F27], # Case map
          0x1F38 => [0x1F30], # Case map
          0x1F39 => [0x1F31], # Case map
          0x1F3A => [0x1F32], # Case map
          0x1F3B => [0x1F33], # Case map
          0x1F3C => [0x1F34], # Case map
          0x1F3D => [0x1F35], # Case map
          0x1F3E => [0x1F36], # Case map
          0x1F3F => [0x1F37], # Case map
          0x1F48 => [0x1F40], # Case map
          0x1F49 => [0x1F41], # Case map
          0x1F4A => [0x1F42], # Case map
          0x1F4B => [0x1F43], # Case map
          0x1F4C => [0x1F44], # Case map
          0x1F4D => [0x1F45], # Case map
          0x1F50 => [0x03C5, 0x0313], # Case map
          0x1F52 => [0x03C5, 0x0313, 0x0300], # Case map
          0x1F54 => [0x03C5, 0x0313, 0x0301], # Case map
          0x1F56 => [0x03C5, 0x0313, 0x0342], # Case map
          0x1F59 => [0x1F51], # Case map
          0x1F5B => [0x1F53], # Case map
          0x1F5D => [0x1F55], # Case map
          0x1F5F => [0x1F57], # Case map
          0x1F68 => [0x1F60], # Case map
          0x1F69 => [0x1F61], # Case map
          0x1F6A => [0x1F62], # Case map
          0x1F6B => [0x1F63], # Case map
          0x1F6C => [0x1F64], # Case map
          0x1F6D => [0x1F65], # Case map
          0x1F6E => [0x1F66], # Case map
          0x1F6F => [0x1F67], # Case map
          0x1F80 => [0x1F00, 0x03B9], # Case map
          0x1F81 => [0x1F01, 0x03B9], # Case map
          0x1F82 => [0x1F02, 0x03B9], # Case map
          0x1F83 => [0x1F03, 0x03B9], # Case map
          0x1F84 => [0x1F04, 0x03B9], # Case map
          0x1F85 => [0x1F05, 0x03B9], # Case map
          0x1F86 => [0x1F06, 0x03B9], # Case map
          0x1F87 => [0x1F07, 0x03B9], # Case map
          0x1F88 => [0x1F00, 0x03B9], # Case map
          0x1F89 => [0x1F01, 0x03B9], # Case map
          0x1F8A => [0x1F02, 0x03B9], # Case map
          0x1F8B => [0x1F03, 0x03B9], # Case map
          0x1F8C => [0x1F04, 0x03B9], # Case map
          0x1F8D => [0x1F05, 0x03B9], # Case map
          0x1F8E => [0x1F06, 0x03B9], # Case map
          0x1F8F => [0x1F07, 0x03B9], # Case map
          0x1F90 => [0x1F20, 0x03B9], # Case map
          0x1F91 => [0x1F21, 0x03B9], # Case map
          0x1F92 => [0x1F22, 0x03B9], # Case map
          0x1F93 => [0x1F23, 0x03B9], # Case map
          0x1F94 => [0x1F24, 0x03B9], # Case map
          0x1F95 => [0x1F25, 0x03B9], # Case map
          0x1F96 => [0x1F26, 0x03B9], # Case map
          0x1F97 => [0x1F27, 0x03B9], # Case map
          0x1F98 => [0x1F20, 0x03B9], # Case map
          0x1F99 => [0x1F21, 0x03B9], # Case map
          0x1F9A => [0x1F22, 0x03B9], # Case map
          0x1F9B => [0x1F23, 0x03B9], # Case map
          0x1F9C => [0x1F24, 0x03B9], # Case map
          0x1F9D => [0x1F25, 0x03B9], # Case map
          0x1F9E => [0x1F26, 0x03B9], # Case map
          0x1F9F => [0x1F27, 0x03B9], # Case map
          0x1FA0 => [0x1F60, 0x03B9], # Case map
          0x1FA1 => [0x1F61, 0x03B9], # Case map
          0x1FA2 => [0x1F62, 0x03B9], # Case map
          0x1FA3 => [0x1F63, 0x03B9], # Case map
          0x1FA4 => [0x1F64, 0x03B9], # Case map
          0x1FA5 => [0x1F65, 0x03B9], # Case map
          0x1FA6 => [0x1F66, 0x03B9], # Case map
          0x1FA7 => [0x1F67, 0x03B9], # Case map
          0x1FA8 => [0x1F60, 0x03B9], # Case map
          0x1FA9 => [0x1F61, 0x03B9], # Case map
          0x1FAA => [0x1F62, 0x03B9], # Case map
          0x1FAB => [0x1F63, 0x03B9], # Case map
          0x1FAC => [0x1F64, 0x03B9], # Case map
          0x1FAD => [0x1F65, 0x03B9], # Case map
          0x1FAE => [0x1F66, 0x03B9], # Case map
          0x1FAF => [0x1F67, 0x03B9], # Case map
          0x1FB2 => [0x1F70, 0x03B9], # Case map
          0x1FB3 => [0x03B1, 0x03B9], # Case map
          0x1FB4 => [0x03AC, 0x03B9], # Case map
          0x1FB6 => [0x03B1, 0x0342], # Case map
          0x1FB7 => [0x03B1, 0x0342, 0x03B9], # Case map
          0x1FB8 => [0x1FB0], # Case map
          0x1FB9 => [0x1FB1], # Case map
          0x1FBA => [0x1F70], # Case map
          0x1FBB => [0x1F71], # Case map
          0x1FBC => [0x03B1, 0x03B9], # Case map
          0x1FBE => [0x03B9], # Case map
          0x1FC2 => [0x1F74, 0x03B9], # Case map
          0x1FC3 => [0x03B7, 0x03B9], # Case map
          0x1FC4 => [0x03AE, 0x03B9], # Case map
          0x1FC6 => [0x03B7, 0x0342], # Case map
          0x1FC7 => [0x03B7, 0x0342, 0x03B9], # Case map
          0x1FC8 => [0x1F72], # Case map
          0x1FC9 => [0x1F73], # Case map
          0x1FCA => [0x1F74], # Case map
          0x1FCB => [0x1F75], # Case map
          0x1FCC => [0x03B7, 0x03B9], # Case map
          0x1FD2 => [0x03B9, 0x0308, 0x0300], # Case map
          0x1FD3 => [0x03B9, 0x0308, 0x0301], # Case map
          0x1FD6 => [0x03B9, 0x0342], # Case map
          0x1FD7 => [0x03B9, 0x0308, 0x0342], # Case map
          0x1FD8 => [0x1FD0], # Case map
          0x1FD9 => [0x1FD1], # Case map
          0x1FDA => [0x1F76], # Case map
          0x1FDB => [0x1F77], # Case map
          0x1FE2 => [0x03C5, 0x0308, 0x0300], # Case map
          0x1FE3 => [0x03C5, 0x0308, 0x0301], # Case map
          0x1FE4 => [0x03C1, 0x0313], # Case map
          0x1FE6 => [0x03C5, 0x0342], # Case map
          0x1FE7 => [0x03C5, 0x0308, 0x0342], # Case map
          0x1FE8 => [0x1FE0], # Case map
          0x1FE9 => [0x1FE1], # Case map
          0x1FEA => [0x1F7A], # Case map
          0x1FEB => [0x1F7B], # Case map
          0x1FEC => [0x1FE5], # Case map
          0x1FF2 => [0x1F7C, 0x03B9], # Case map
          0x1FF3 => [0x03C9, 0x03B9], # Case map
          0x1FF4 => [0x03CE, 0x03B9], # Case map
          0x1FF6 => [0x03C9, 0x0342], # Case map
          0x1FF7 => [0x03C9, 0x0342, 0x03B9], # Case map
          0x1FF8 => [0x1F78], # Case map
          0x1FF9 => [0x1F79], # Case map
          0x1FFA => [0x1F7C], # Case map
          0x1FFB => [0x1F7D], # Case map
          0x1FFC => [0x03C9, 0x03B9], # Case map
          0x2126 => [0x03C9], # Case map
          0x212A => [0x006B], # Case map
          0x212B => [0x00E5], # Case map
          0x2160 => [0x2170], # Case map
          0x2161 => [0x2171], # Case map
          0x2162 => [0x2172], # Case map
          0x2163 => [0x2173], # Case map
          0x2164 => [0x2174], # Case map
          0x2165 => [0x2175], # Case map
          0x2166 => [0x2176], # Case map
          0x2167 => [0x2177], # Case map
          0x2168 => [0x2178], # Case map
          0x2169 => [0x2179], # Case map
          0x216A => [0x217A], # Case map
          0x216B => [0x217B], # Case map
          0x216C => [0x217C], # Case map
          0x216D => [0x217D], # Case map
          0x216E => [0x217E], # Case map
          0x216F => [0x217F], # Case map
          0x24B6 => [0x24D0], # Case map
          0x24B7 => [0x24D1], # Case map
          0x24B8 => [0x24D2], # Case map
          0x24B9 => [0x24D3], # Case map
          0x24BA => [0x24D4], # Case map
          0x24BB => [0x24D5], # Case map
          0x24BC => [0x24D6], # Case map
          0x24BD => [0x24D7], # Case map
          0x24BE => [0x24D8], # Case map
          0x24BF => [0x24D9], # Case map
          0x24C0 => [0x24DA], # Case map
          0x24C1 => [0x24DB], # Case map
          0x24C2 => [0x24DC], # Case map
          0x24C3 => [0x24DD], # Case map
          0x24C4 => [0x24DE], # Case map
          0x24C5 => [0x24DF], # Case map
          0x24C6 => [0x24E0], # Case map
          0x24C7 => [0x24E1], # Case map
          0x24C8 => [0x24E2], # Case map
          0x24C9 => [0x24E3], # Case map
          0x24CA => [0x24E4], # Case map
          0x24CB => [0x24E5], # Case map
          0x24CC => [0x24E6], # Case map
          0x24CD => [0x24E7], # Case map
          0x24CE => [0x24E8], # Case map
          0x24CF => [0x24E9], # Case map
          0xFB00 => [0x0066, 0x0066], # Case map
          0xFB01 => [0x0066, 0x0069], # Case map
          0xFB02 => [0x0066, 0x006C], # Case map
          0xFB03 => [0x0066, 0x0066, 0x0069], # Case map
          0xFB04 => [0x0066, 0x0066, 0x006C], # Case map
          0xFB05 => [0x0073, 0x0074], # Case map
          0xFB06 => [0x0073, 0x0074], # Case map
          0xFB13 => [0x0574, 0x0576], # Case map
          0xFB14 => [0x0574, 0x0565], # Case map
          0xFB15 => [0x0574, 0x056B], # Case map
          0xFB16 => [0x057E, 0x0576], # Case map
          0xFB17 => [0x0574, 0x056D], # Case map
          0xFF21 => [0xFF41], # Case map
          0xFF22 => [0xFF42], # Case map
          0xFF23 => [0xFF43], # Case map
          0xFF24 => [0xFF44], # Case map
          0xFF25 => [0xFF45], # Case map
          0xFF26 => [0xFF46], # Case map
          0xFF27 => [0xFF47], # Case map
          0xFF28 => [0xFF48], # Case map
          0xFF29 => [0xFF49], # Case map
          0xFF2A => [0xFF4A], # Case map
          0xFF2B => [0xFF4B], # Case map
          0xFF2C => [0xFF4C], # Case map
          0xFF2D => [0xFF4D], # Case map
          0xFF2E => [0xFF4E], # Case map
          0xFF2F => [0xFF4F], # Case map
          0xFF30 => [0xFF50], # Case map
          0xFF31 => [0xFF51], # Case map
          0xFF32 => [0xFF52], # Case map
          0xFF33 => [0xFF53], # Case map
          0xFF34 => [0xFF54], # Case map
          0xFF35 => [0xFF55], # Case map
          0xFF36 => [0xFF56], # Case map
          0xFF37 => [0xFF57], # Case map
          0xFF38 => [0xFF58], # Case map
          0xFF39 => [0xFF59], # Case map
          0xFF3A => [0xFF5A], # Case map
          0x10400 => [0x10428], # Case map
          0x10401 => [0x10429], # Case map
          0x10402 => [0x1042A], # Case map
          0x10403 => [0x1042B], # Case map
          0x10404 => [0x1042C], # Case map
          0x10405 => [0x1042D], # Case map
          0x10406 => [0x1042E], # Case map
          0x10407 => [0x1042F], # Case map
          0x10408 => [0x10430], # Case map
          0x10409 => [0x10431], # Case map
          0x1040A => [0x10432], # Case map
          0x1040B => [0x10433], # Case map
          0x1040C => [0x10434], # Case map
          0x1040D => [0x10435], # Case map
          0x1040E => [0x10436], # Case map
          0x1040F => [0x10437], # Case map
          0x10410 => [0x10438], # Case map
          0x10411 => [0x10439], # Case map
          0x10412 => [0x1043A], # Case map
          0x10413 => [0x1043B], # Case map
          0x10414 => [0x1043C], # Case map
          0x10415 => [0x1043D], # Case map
          0x10416 => [0x1043E], # Case map
          0x10417 => [0x1043F], # Case map
          0x10418 => [0x10440], # Case map
          0x10419 => [0x10441], # Case map
          0x1041A => [0x10442], # Case map
          0x1041B => [0x10443], # Case map
          0x1041C => [0x10444], # Case map
          0x1041D => [0x10445], # Case map
          0x1041E => [0x10446], # Case map
          0x1041F => [0x10447], # Case map
          0x10420 => [0x10448], # Case map
          0x10421 => [0x10449], # Case map
          0x10422 => [0x1044A], # Case map
          0x10423 => [0x1044B], # Case map
          0x10424 => [0x1044C], # Case map
          0x10425 => [0x1044D], # Case map
        }.freeze

        # Table C1.1 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        C1_1 = [
          0x0020..0x0020, # SPACE
        ]

        # Table C1.2 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        C1_2 = [
          0x00A0..0x00A0, # NO-BREAK SPACE
          0x1680..0x1680, # OGHAM SPACE MARK
          0x2000..0x2000, # EN QUAD
          0x2001..0x2001, # EM QUAD
          0x2002..0x2002, # EN SPACE
          0x2003..0x2003, # EM SPACE
          0x2004..0x2004, # THREE-PER-EM SPACE
          0x2005..0x2005, # FOUR-PER-EM SPACE
          0x2006..0x2006, # SIX-PER-EM SPACE
          0x2007..0x2007, # FIGURE SPACE
          0x2008..0x2008, # PUNCTUATION SPACE
          0x2009..0x2009, # THIN SPACE
          0x200A..0x200A, # HAIR SPACE
          0x200B..0x200B, # ZERO WIDTH SPACE
          0x202F..0x202F, # NARROW NO-BREAK SPACE
          0x205F..0x205F, # MEDIUM MATHEMATICAL SPACE
          0x3000..0x3000, # IDEOGRAPHIC SPACE
        ].freeze

        # Table C2.1 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        C2_1 = [
          0x0000..0x001F, # [CONTROL CHARACTERS]
          0x007F..0x007F, # DELETE
        ].freeze

        # Table C2.2 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        C2_2 = [
          0x0080..0x009F, # [CONTROL CHARACTERS]
          0x06DD..0x06DD, # ARABIC END OF AYAH
          0x070F..0x070F, # SYRIAC ABBREVIATION MARK
          0x180E..0x180E, # MONGOLIAN VOWEL SEPARATOR
          0x200C..0x200C, # ZERO WIDTH NON-JOINER
          0x200D..0x200D, # ZERO WIDTH JOINER
          0x2028..0x2028, # LINE SEPARATOR
          0x2029..0x2029, # PARAGRAPH SEPARATOR
          0x2060..0x2060, # WORD JOINER
          0x2061..0x2061, # FUNCTION APPLICATION
          0x2062..0x2062, # INVISIBLE TIMES
          0x2063..0x2063, # INVISIBLE SEPARATOR
          0x206A..0x206F, # [CONTROL CHARACTERS]
          0xFEFF..0xFEFF, # ZERO WIDTH NO-BREAK SPACE
          0xFFF9..0xFFFC, # [CONTROL CHARACTERS]
          0x1D173..0x1D17A, # [MUSICAL CONTROL CHARACTERS]
        ].freeze

        # Table C3 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        C3 = [
          0xE000..0xF8FF, # [PRIVATE USE, PLANE 0]
          0xF0000..0xFFFFD, # [PRIVATE USE, PLANE 15]
          0x100000..0x10FFFD, # [PRIVATE USE, PLANE 16]
        ].freeze

        # Table C4 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        C4 = [
          0xFDD0..0xFDEF, # [NONCHARACTER CODE POINTS]
          0xFFFE..0xFFFF, # [NONCHARACTER CODE POINTS]
          0x1FFFE..0x1FFFF, # [NONCHARACTER CODE POINTS]
          0x2FFFE..0x2FFFF, # [NONCHARACTER CODE POINTS]
          0x3FFFE..0x3FFFF, # [NONCHARACTER CODE POINTS]
          0x4FFFE..0x4FFFF, # [NONCHARACTER CODE POINTS]
          0x5FFFE..0x5FFFF, # [NONCHARACTER CODE POINTS]
          0x6FFFE..0x6FFFF, # [NONCHARACTER CODE POINTS]
          0x7FFFE..0x7FFFF, # [NONCHARACTER CODE POINTS]
          0x8FFFE..0x8FFFF, # [NONCHARACTER CODE POINTS]
          0x9FFFE..0x9FFFF, # [NONCHARACTER CODE POINTS]
          0xAFFFE..0xAFFFF, # [NONCHARACTER CODE POINTS]
          0xBFFFE..0xBFFFF, # [NONCHARACTER CODE POINTS]
          0xCFFFE..0xCFFFF, # [NONCHARACTER CODE POINTS]
          0xDFFFE..0xDFFFF, # [NONCHARACTER CODE POINTS]
          0xEFFFE..0xEFFFF, # [NONCHARACTER CODE POINTS]
          0xFFFFE..0xFFFFF, # [NONCHARACTER CODE POINTS]
          0x10FFFE..0x10FFFF, # [NONCHARACTER CODE POINTS]
        ].freeze

        # Table C5 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        C5 = [
          0xD800..0xDFFF, # [SURROGATE CODES]
        ].freeze

        # Table C6 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        C6 = [
          0xFFF9..0xFFF9, # INTERLINEAR ANNOTATION ANCHOR
          0xFFFA..0xFFFA, # INTERLINEAR ANNOTATION SEPARATOR
          0xFFFB..0xFFFB, # INTERLINEAR ANNOTATION TERMINATOR
          0xFFFC..0xFFFC, # OBJECT REPLACEMENT CHARACTER
          0xFFFD..0xFFFD, # REPLACEMENT CHARACTER
        ].freeze

        # Table C7 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        C7 = [
          0x2FF0..0x2FFB, # [IDEOGRAPHIC DESCRIPTION CHARACTERS]
        ].freeze

        # Table C8 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        C8 = [
          0x0340..0x0340, # COMBINING GRAVE TONE MARK
          0x0341..0x0341, # COMBINING ACUTE TONE MARK
          0x200E..0x200E, # LEFT-TO-RIGHT MARK
          0x200F..0x200F, # RIGHT-TO-LEFT MARK
          0x202A..0x202A, # LEFT-TO-RIGHT EMBEDDING
          0x202B..0x202B, # RIGHT-TO-LEFT EMBEDDING
          0x202C..0x202C, # POP DIRECTIONAL FORMATTING
          0x202D..0x202D, # LEFT-TO-RIGHT OVERRIDE
          0x202E..0x202E, # RIGHT-TO-LEFT OVERRIDE
          0x206A..0x206A, # INHIBIT SYMMETRIC SWAPPING
          0x206B..0x206B, # ACTIVATE SYMMETRIC SWAPPING
          0x206C..0x206C, # INHIBIT ARABIC FORM SHAPING
          0x206D..0x206D, # ACTIVATE ARABIC FORM SHAPING
          0x206E..0x206E, # NATIONAL DIGIT SHAPES
          0x206F..0x206F, # NOMINAL DIGIT SHAPES
        ].freeze

        # Table C9 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        C9 = [
          0xE0001..0xE0001, # LANGUAGE TAG
          0xE0020..0xE007F, # [TAGGING CHARACTERS]
        ].freeze

        # Table D1 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        D1 = [
          0x05BE..0x05BE,
          0x05C0..0x05C0,
          0x05C3..0x05C3,
          0x05D0..0x05EA,
          0x05F0..0x05F4,
          0x061B..0x061B,
          0x061F..0x061F,
          0x0621..0x063A,
          0x0640..0x064A,
          0x066D..0x066F,
          0x0671..0x06D5,
          0x06DD..0x06DD,
          0x06E5..0x06E6,
          0x06FA..0x06FE,
          0x0700..0x070D,
          0x0710..0x0710,
          0x0712..0x072C,
          0x0780..0x07A5,
          0x07B1..0x07B1,
          0x200F..0x200F,
          0xFB1D..0xFB1D,
          0xFB1F..0xFB28,
          0xFB2A..0xFB36,
          0xFB38..0xFB3C,
          0xFB3E..0xFB3E,
          0xFB40..0xFB41,
          0xFB43..0xFB44,
          0xFB46..0xFBB1,
          0xFBD3..0xFD3D,
          0xFD50..0xFD8F,
          0xFD92..0xFDC7,
          0xFDF0..0xFDFC,
          0xFE70..0xFE74,
          0xFE76..0xFEFC,
        ].freeze

        # Table D2 as defined by RFC 3454 (string preparation).
        #
        # @since 2.6.0
        D2 = [
          0x0041..0x005A,
          0x0061..0x007A,
          0x00AA..0x00AA,
          0x00B5..0x00B5,
          0x00BA..0x00BA,
          0x00C0..0x00D6,
          0x00D8..0x00F6,
          0x00F8..0x0220,
          0x0222..0x0233,
          0x0250..0x02AD,
          0x02B0..0x02B8,
          0x02BB..0x02C1,
          0x02D0..0x02D1,
          0x02E0..0x02E4,
          0x02EE..0x02EE,
          0x037A..0x037A,
          0x0386..0x0386,
          0x0388..0x038A,
          0x038C..0x038C,
          0x038E..0x03A1,
          0x03A3..0x03CE,
          0x03D0..0x03F5,
          0x0400..0x0482,
          0x048A..0x04CE,
          0x04D0..0x04F5,
          0x04F8..0x04F9,
          0x0500..0x050F,
          0x0531..0x0556,
          0x0559..0x055F,
          0x0561..0x0587,
          0x0589..0x0589,
          0x0903..0x0903,
          0x0905..0x0939,
          0x093D..0x0940,
          0x0949..0x094C,
          0x0950..0x0950,
          0x0958..0x0961,
          0x0964..0x0970,
          0x0982..0x0983,
          0x0985..0x098C,
          0x098F..0x0990,
          0x0993..0x09A8,
          0x09AA..0x09B0,
          0x09B2..0x09B2,
          0x09B6..0x09B9,
          0x09BE..0x09C0,
          0x09C7..0x09C8,
          0x09CB..0x09CC,
          0x09D7..0x09D7,
          0x09DC..0x09DD,
          0x09DF..0x09E1,
          0x09E6..0x09F1,
          0x09F4..0x09FA,
          0x0A05..0x0A0A,
          0x0A0F..0x0A10,
          0x0A13..0x0A28,
          0x0A2A..0x0A30,
          0x0A32..0x0A33,
          0x0A35..0x0A36,
          0x0A38..0x0A39,
          0x0A3E..0x0A40,
          0x0A59..0x0A5C,
          0x0A5E..0x0A5E,
          0x0A66..0x0A6F,
          0x0A72..0x0A74,
          0x0A83..0x0A83,
          0x0A85..0x0A8B,
          0x0A8D..0x0A8D,
          0x0A8F..0x0A91,
          0x0A93..0x0AA8,
          0x0AAA..0x0AB0,
          0x0AB2..0x0AB3,
          0x0AB5..0x0AB9,
          0x0ABD..0x0AC0,
          0x0AC9..0x0AC9,
          0x0ACB..0x0ACC,
          0x0AD0..0x0AD0,
          0x0AE0..0x0AE0,
          0x0AE6..0x0AEF,
          0x0B02..0x0B03,
          0x0B05..0x0B0C,
          0x0B0F..0x0B10,
          0x0B13..0x0B28,
          0x0B2A..0x0B30,
          0x0B32..0x0B33,
          0x0B36..0x0B39,
          0x0B3D..0x0B3E,
          0x0B40..0x0B40,
          0x0B47..0x0B48,
          0x0B4B..0x0B4C,
          0x0B57..0x0B57,
          0x0B5C..0x0B5D,
          0x0B5F..0x0B61,
          0x0B66..0x0B70,
          0x0B83..0x0B83,
          0x0B85..0x0B8A,
          0x0B8E..0x0B90,
          0x0B92..0x0B95,
          0x0B99..0x0B9A,
          0x0B9C..0x0B9C,
          0x0B9E..0x0B9F,
          0x0BA3..0x0BA4,
          0x0BA8..0x0BAA,
          0x0BAE..0x0BB5,
          0x0BB7..0x0BB9,
          0x0BBE..0x0BBF,
          0x0BC1..0x0BC2,
          0x0BC6..0x0BC8,
          0x0BCA..0x0BCC,
          0x0BD7..0x0BD7,
          0x0BE7..0x0BF2,
          0x0C01..0x0C03,
          0x0C05..0x0C0C,
          0x0C0E..0x0C10,
          0x0C12..0x0C28,
          0x0C2A..0x0C33,
          0x0C35..0x0C39,
          0x0C41..0x0C44,
          0x0C60..0x0C61,
          0x0C66..0x0C6F,
          0x0C82..0x0C83,
          0x0C85..0x0C8C,
          0x0C8E..0x0C90,
          0x0C92..0x0CA8,
          0x0CAA..0x0CB3,
          0x0CB5..0x0CB9,
          0x0CBE..0x0CBE,
          0x0CC0..0x0CC4,
          0x0CC7..0x0CC8,
          0x0CCA..0x0CCB,
          0x0CD5..0x0CD6,
          0x0CDE..0x0CDE,
          0x0CE0..0x0CE1,
          0x0CE6..0x0CEF,
          0x0D02..0x0D03,
          0x0D05..0x0D0C,
          0x0D0E..0x0D10,
          0x0D12..0x0D28,
          0x0D2A..0x0D39,
          0x0D3E..0x0D40,
          0x0D46..0x0D48,
          0x0D4A..0x0D4C,
          0x0D57..0x0D57,
          0x0D60..0x0D61,
          0x0D66..0x0D6F,
          0x0D82..0x0D83,
          0x0D85..0x0D96,
          0x0D9A..0x0DB1,
          0x0DB3..0x0DBB,
          0x0DBD..0x0DBD,
          0x0DC0..0x0DC6,
          0x0DCF..0x0DD1,
          0x0DD8..0x0DDF,
          0x0DF2..0x0DF4,
          0x0E01..0x0E30,
          0x0E32..0x0E33,
          0x0E40..0x0E46,
          0x0E4F..0x0E5B,
          0x0E81..0x0E82,
          0x0E84..0x0E84,
          0x0E87..0x0E88,
          0x0E8A..0x0E8A,
          0x0E8D..0x0E8D,
          0x0E94..0x0E97,
          0x0E99..0x0E9F,
          0x0EA1..0x0EA3,
          0x0EA5..0x0EA5,
          0x0EA7..0x0EA7,
          0x0EAA..0x0EAB,
          0x0EAD..0x0EB0,
          0x0EB2..0x0EB3,
          0x0EBD..0x0EBD,
          0x0EC0..0x0EC4,
          0x0EC6..0x0EC6,
          0x0ED0..0x0ED9,
          0x0EDC..0x0EDD,
          0x0F00..0x0F17,
          0x0F1A..0x0F34,
          0x0F36..0x0F36,
          0x0F38..0x0F38,
          0x0F3E..0x0F47,
          0x0F49..0x0F6A,
          0x0F7F..0x0F7F,
          0x0F85..0x0F85,
          0x0F88..0x0F8B,
          0x0FBE..0x0FC5,
          0x0FC7..0x0FCC,
          0x0FCF..0x0FCF,
          0x1000..0x1021,
          0x1023..0x1027,
          0x1029..0x102A,
          0x102C..0x102C,
          0x1031..0x1031,
          0x1038..0x1038,
          0x1040..0x1057,
          0x10A0..0x10C5,
          0x10D0..0x10F8,
          0x10FB..0x10FB,
          0x1100..0x1159,
          0x115F..0x11A2,
          0x11A8..0x11F9,
          0x1200..0x1206,
          0x1208..0x1246,
          0x1248..0x1248,
          0x124A..0x124D,
          0x1250..0x1256,
          0x1258..0x1258,
          0x125A..0x125D,
          0x1260..0x1286,
          0x1288..0x1288,
          0x128A..0x128D,
          0x1290..0x12AE,
          0x12B0..0x12B0,
          0x12B2..0x12B5,
          0x12B8..0x12BE,
          0x12C0..0x12C0,
          0x12C2..0x12C5,
          0x12C8..0x12CE,
          0x12D0..0x12D6,
          0x12D8..0x12EE,
          0x12F0..0x130E,
          0x1310..0x1310,
          0x1312..0x1315,
          0x1318..0x131E,
          0x1320..0x1346,
          0x1348..0x135A,
          0x1361..0x137C,
          0x13A0..0x13F4,
          0x1401..0x1676,
          0x1681..0x169A,
          0x16A0..0x16F0,
          0x1700..0x170C,
          0x170E..0x1711,
          0x1720..0x1731,
          0x1735..0x1736,
          0x1740..0x1751,
          0x1760..0x176C,
          0x176E..0x1770,
          0x1780..0x17B6,
          0x17BE..0x17C5,
          0x17C7..0x17C8,
          0x17D4..0x17DA,
          0x17DC..0x17DC,
          0x17E0..0x17E9,
          0x1810..0x1819,
          0x1820..0x1877,
          0x1880..0x18A8,
          0x1E00..0x1E9B,
          0x1EA0..0x1EF9,
          0x1F00..0x1F15,
          0x1F18..0x1F1D,
          0x1F20..0x1F45,
          0x1F48..0x1F4D,
          0x1F50..0x1F57,
          0x1F59..0x1F59,
          0x1F5B..0x1F5B,
          0x1F5D..0x1F5D,
          0x1F5F..0x1F7D,
          0x1F80..0x1FB4,
          0x1FB6..0x1FBC,
          0x1FBE..0x1FBE,
          0x1FC2..0x1FC4,
          0x1FC6..0x1FCC,
          0x1FD0..0x1FD3,
          0x1FD6..0x1FDB,
          0x1FE0..0x1FEC,
          0x1FF2..0x1FF4,
          0x1FF6..0x1FFC,
          0x200E..0x200E,
          0x2071..0x2071,
          0x207F..0x207F,
          0x2102..0x2102,
          0x2107..0x2107,
          0x210A..0x2113,
          0x2115..0x2115,
          0x2119..0x211D,
          0x2124..0x2124,
          0x2126..0x2126,
          0x2128..0x2128,
          0x212A..0x212D,
          0x212F..0x2131,
          0x2133..0x2139,
          0x213D..0x213F,
          0x2145..0x2149,
          0x2160..0x2183,
          0x2336..0x237A,
          0x2395..0x2395,
          0x249C..0x24E9,
          0x3005..0x3007,
          0x3021..0x3029,
          0x3031..0x3035,
          0x3038..0x303C,
          0x3041..0x3096,
          0x309D..0x309F,
          0x30A1..0x30FA,
          0x30FC..0x30FF,
          0x3105..0x312C,
          0x3131..0x318E,
          0x3190..0x31B7,
          0x31F0..0x321C,
          0x3220..0x3243,
          0x3260..0x327B,
          0x327F..0x32B0,
          0x32C0..0x32CB,
          0x32D0..0x32FE,
          0x3300..0x3376,
          0x337B..0x33DD,
          0x33E0..0x33FE,
          0x3400..0x4DB5,
          0x4E00..0x9FA5,
          0xA000..0xA48C,
          0xAC00..0xD7A3,
          0xD800..0xFA2D,
          0xFA30..0xFA6A,
          0xFB00..0xFB06,
          0xFB13..0xFB17,
          0xFF21..0xFF3A,
          0xFF41..0xFF5A,
          0xFF66..0xFFBE,
          0xFFC2..0xFFC7,
          0xFFCA..0xFFCF,
          0xFFD2..0xFFD7,
          0xFFDA..0xFFDC,
          0x10300..0x1031E,
          0x10320..0x10323,
          0x10330..0x1034A,
          0x10400..0x10425,
          0x10428..0x1044D,
          0x1D000..0x1D0F5,
          0x1D100..0x1D126,
          0x1D12A..0x1D166,
          0x1D16A..0x1D172,
          0x1D183..0x1D184,
          0x1D18C..0x1D1A9,
          0x1D1AE..0x1D1DD,
          0x1D400..0x1D454,
          0x1D456..0x1D49C,
          0x1D49E..0x1D49F,
          0x1D4A2..0x1D4A2,
          0x1D4A5..0x1D4A6,
          0x1D4A9..0x1D4AC,
          0x1D4AE..0x1D4B9,
          0x1D4BB..0x1D4BB,
          0x1D4BD..0x1D4C0,
          0x1D4C2..0x1D4C3,
          0x1D4C5..0x1D505,
          0x1D507..0x1D50A,
          0x1D50D..0x1D514,
          0x1D516..0x1D51C,
          0x1D51E..0x1D539,
          0x1D53B..0x1D53E,
          0x1D540..0x1D544,
          0x1D546..0x1D546,
          0x1D54A..0x1D550,
          0x1D552..0x1D6A3,
          0x1D6A8..0x1D7C9,
          0x20000..0x2A6D6,
          0x2F800..0x2FA1D,
          0xF0000..0xFFFFD,
          0x100000..0x10FFFD,
        ].freeze
      end
    end
  end
end
